/**			       
*
*	Program Name:	Tftp Loading Module
*
*	Filename:	tftputil.c
*
*	$Log:   /b/gregs/i960/tcpip/tftp/tftputil.c_v  $
 * 
 *    Rev 1.3   12 Oct 1993 10:45:14   franks
 * No change.
 * 
 *    Rev 1.2   29 Sep 1993 10:38:16   franks
 * No change.
 * 
 *    Rev 1.1   30 Jul 1993 13:56:10   franks
 * No change.
 * 
 *    Rev 1.0   14 Jul 1993 10:18:50   gregs
 * Initial revision.
 * 
 *    Rev 1.3   16 Jun 1992 17:43:34   vinay
 * changed back to rev 1.1
 * 
 *    Rev 1.2   16 Jun 1992 15:58:44   vinay
 * Changed printf to include Errorlog function. This is done only for error msgs
 * 
 *    Rev 1.1   13 May 1992 11:00:52   pvcs
 * 
 *    Rev 1.0   30 Mar 1992 17:25:16   pvcs
 * Initial revision.
*
*	Creation Date:	7.15.91
*
*	Date:
*
*	Version:	1.0
*
*	Programmers:
*
*	Modifications:	
*
*	Comments:	This file contains utilities functions for 
*			the tftp loader
*			-	UseBootRecord
*			-	EnterBootRecord
*			-	AskYesNo
*			-	AskIP
*			-	AskForBaud
*			-	CheckImage
*
*
*	Copyright (c) 1991 by Hughes LAN Systems
*
**/
#include <types.h>
#include <nim960h.h>
#include <netbuf.h>
#include <udp.h>
#include <tftpboot.h>



/*
 * name		UseBootRecord	
 *
 * synopsis	int UseBootRecord(boot)
 *		BOOT_BLOCK	*boot; <<	the boot record
 *
 * description	It displays the current setting of the boot record, such
 *		as my ip address, server ip ...etc. It then asks the user
 *		if the user wants to use these parameters or the user
 *		wants to enter new parameters.
 *
 * returns	0	user does not want to use the current boot record
 *		otherwise	use the boot record
 */

int UseBootRecord(boot)
BOOT_BLOCK	*boot;

	{

	printf("The following boot record has been found:-\n\n");
	printf("This station's IP address: %s\n", inet_ntoa(boot->MyIP));
	printf("This sub-net's IP net-mask: %s\n", inet_ntoa(boot->NetMask));
	printf("Server station's IP address: %s\n", inet_ntoa(boot->ServerIP));
	if ((boot->MyIP & boot->NetMask) != 
		(boot->ServerIP & boot->NetMask))
		{
		/*
		 *	server is not on the same network 
		 */
		printf("IP router address is %s\n", 
			inet_ntoa(boot->RouterIP));
		}
	printf("Configuration file to be loaded: %s\n", boot->FileName);
	return AskYesNo("Do you want to use these boot records([y]/n) ==>");
	}
/*
 * name		EnterBootRecord	- ask user to enter the boot record
 *
 * synopsis	void EnterBootRecord(boot)
 *		BOOT_BLOCK	*boot; >>	boot record to be filled in
 *
 * description	It asks the user to enter the boot parameters such as
 *		my ip, server ip and file name to be downloaded.  The
 *		parameters will be returned in the boot record "boot".
 *
 * returns	nothing
 */

void EnterBootRecord(boot)
BOOT_BLOCK	*boot;

	{
	int	confirmed = 0;
	char	filename[TFTP_FILE_LENGTH];

	while (confirmed == 0)
		{
		/*
		 *	ask user to enter information
		 */
		AskIP("Enter this stations's IP address", 
			&boot->MyIP);
		AskIP("Enter this sub-net's IP net-mask", 
			&boot->NetMask);
		AskIP("Enter server station's IP address", 
			&boot->ServerIP);
		if ((boot->MyIP & boot->NetMask) != 
			(boot->ServerIP & boot->NetMask))
			{
			/*
			 *	server is not on the same network 
			 *	need to ask for the gateway's ip address
			 */
			AskIP("Enter this sub-net's IP router address",
				&boot->RouterIP);
			}
		for (;;)
			{
			printf("Enter the configuration file to be loaded (%s) ==>", 
				boot->FileName[0] != '\0' ? boot->FileName : 
				"NULL");
			gets(filename);
			if (filename[0] != '\0')
				{
				strcpy(boot->FileName, filename);
				break;
				}
			else if (boot->FileName[0] != '\0')
				break;
			}
		confirmed = AskYesNo("Confirm ([y]/n) ==>");
		}
	}
#ifdef kwok
/*
 * name		AskYesNo	- prompt the user to enter y or n
 *
 * synopsis	AskYesNo(char *message)
 *		char	*message;<<	to prompt the user 
 *
 * description	It displays the "message" and wait for the user to enter
 *		'y'('Y') or 'n'('N').  If the user presses 'enter', 
 *		then the default answer is 'y'.
 *		If the user enters any other characters, the
 *		sequence will be repeated.
 *
 * returns	0		- user enter 'n'
 *		otherwise	- user enter 'y'
 */

AskYesNo(char *message)

	{
	char	answer = '\0';
	int	tmp;
	int	i;

	for (;;)
		{
		printf("%s", message);
		i = scanf("%c%c", &answer, &tmp);
		if (i == 0)
			answer = 'y';
		answer = tolower(answer);
		if (i <= 1)
			{
			if (answer == 'y' || answer == 'n')
				break;
			}
		printf("Syntax Error: Please try again\n");
		}
	return answer == 'y' ? 1 : 0;
	}

/*
 * name		AskIP	- ask the user to enter an ip address
 *
 * synopsis	AskIP(message, ip)
 *		char	*message; <<	displayed to prompt the user
 *		in_name	*ip;>>		the ip address entered by the user
 *					will be returned in here.
 *
 * description	It displays the "message" to prompt the user to enter
 *		an ip address in the format of (dd.dd.dd.dd).  The
 *		ip address will be retured in "ip", which is ready to 
 *		be used without any further formating. The default value
 *		of ip will be displayed and if the user press ENTER,
 *		no new ip will be returned.
 *
 * returns	nothing
 */

void AskIP(char *message, in_name *ip)

	{
	uint	tmp = -1;	/*	tmp value of ip	*/
	char	buffer[80];

	for (;;)
		{
		printf("%s (%s) ==>", message, inet_ntoa(*ip));
		gets(buffer);
		/*
		 *	The user wants to take the default value.
		 */
		if (buffer[0] == '\0' || (tmp = inet_addr(buffer)) != -1)
			{
			if (tmp != -1)
				*ip = tmp;
			break;
			}
		printf("Syntax Error: Please try again\n");
		}
	}

#endif

/*
 * name		AskForBaud	- ask the user to enter the baud rate
 *
 * synopsis	int AskForBaud(void)
 *
 * description	It prompts the user to choose a baud rate for SLIP.
 *		The choices are :-
 *		1 = 1200, 2 = 2400, 3 = 4800, 4 = 9600
 *		5 = 19200, 6 = 38400.
 *
 * returns	baud rate
 */

int AskForBaud()
	{
	int	baud;	/* 1 = 1200, 2 = 2400, 3 = 4800, 4 = 9600
			 * 5 = 19200 and 6 = 38400
			 */
	int	argc;	/* how many arguments have been entered	*/
	int	c;
	static	int	BaudRate[] = 
				{
				1200, 
				2400, 
				4800,
				9600,
				19200,
				38400
				};

	for (;;)
		{
		printf("Choose a baud rate for SLIP : 1 (1200), 2 (2400)\n");
		printf("3 (4800), 4 (9600), 5 (19200) or 6 (38400) ==>");
		if ((argc = scanf("%d%c", &baud, &c)) == 1 &&
			1 <= baud && baud <= 6)
			{
			break;
			}
		if (argc != 1)
			printf("Syntax Error: Please try again\n");
		else
			printf("Choice out of range: Please try again\n");
		}
	return BaudRate[baud - 1];
	}

/*
 * name		CheckImage
 *
 * synopsis	CheckImage(LoadAddress, FileLength)
 *		byte	*LoadAddress; <<	starting address
 *		ulong	FileLength; <<		to be checked
 *
 * description	It checks if the file in memory address "LoadAddress"
 *		with length "FileLength" is a valid nim960 file format.
 *
 * returns	TRUE	file is a valid nim960 file 
 *		FALSE	otherwise
 */

CheckImage(byte *LoadAddress, ulong FileLength)

	{
	NIM960_HDR	*header;
	unsigned short	checksum;

	/*
	 *	Check that the file in intact
	 *	1	Does FileLength matches the file length field in
	 *		the module header
	 *	2	Is file checksum ok
	 *	3	Is the file a nim960 file
	 */
	header = (NIM960_HDR *)LoadAddress;
	if (header->signature != NIM960_MAGIC)
	{
		printf("Error: Invalid file format\n");
	}
	else if (header->FileLength != FileLength)
		{
		printf("Error: Bad file length %d, received %d\n", 
			header->FileLength, FileLength);
		}
	else if ((checksum = cksum((ushort *)LoadAddress, FileLength >> 1)) != 0xffff)
		{
		if (checksum == 0)
			return TRUE;	
		printf("Error: File checksum error (%x)\n", checksum);
		}
	else
	    	return TRUE;
	return FALSE;
	}
