/*
 *	This file contains the EEPROM load and update functions.
 *
 *	Written by 	Ramakrishna Rao
 *	modified by	K Kong	10.24.91
 *			Adding the InitEeprom and init the markers.
 *
 *	$Log:   /b/gregs/i960/util/eeprom.c_v  $
 * 
 *    Rev 1.2   12 Oct 1993 10:38:18   franks
 * No change.
 * 
 *    Rev 1.1   29 Sep 1993 10:27:00   franks
 * No change.
 * 
 *    Rev 1.0   14 Jul 1993 10:24:54   gregs
 * Initial revision.
 * 
 *    Rev 1.1   15 Apr 1992 13:58:34   kwok
 * EepromDirty should be of type volatile.
 * 
 *    Rev 1.0   30 Mar 1992 17:08:56   pvcs
 * Initial revision.
 *
 *	Copyright (c) by 1992 by Hughes LAN Systems
 */
#include <types.h>
#include <krnl.h>
#include <eeprom.h>

EEP_HDR		Eeprom_header;
volatile u_int		EepromDirty;

static byte	*EepromOffset;
static byte	*EepromBufadr;

static void	*EepromAddr;
static int	EepromSize;

/*
 * name		InitEeprom
 *
 * synopsis	InitEeprom(eeprom. size)
 *		void	*eeprom; <<	starting address of eeprom
 *		int	size; <<	of eeprom in bytes
 *
 * description	This function should be called before the eeprom 
 *		primitives are called. It sets up the eeprom header if
 *		this is the first time the eeprom is being used. 
 *
 * returns	EEPROM_AOK	- eeprom is ready to be used
 *		error code	otherwise
 */

InitEeprom(void *eeprom, int size)

	{
	EepromAddr = eeprom;
	EepromSize = size;
	if (GetEeprom(EEP_HDR_ADDR, &Eeprom_header, EEP_HDR_SIZE) != EEPROM_AOK)
		{
		memset(&Eeprom_header, 0, sizeof(EEP_HDR));
		Eeprom_header.eep_marker = EEP_MARKER;
		Eeprom_header.eep_recs[0].offset = EEP_HDR_ADDR;
		Eeprom_header.eep_recs[0].size = EEP_HDR_SIZE;
		/*
		 *	We cannot init. (i.e. call PutEeprom()) in here
		 *	as the kernel may not be active yet!
		 *	We set the EepromHeaderValid flag to signal
		 *	that eeprom is not ready.
		 */
		return EEPROM_NOSUCH;
		}
	return EEPROM_AOK;
	}
		

GetEeprom(offset,bufadr, size)
u_int	offset;
byte	*bufadr;
int	size;
{
	int	i;
	int	found;
	u_short *cksump;
	byte	*eep;

	if(((int)offset & 0x1f) || (size & 0x1f)) {
        printf("\nEEPROM_ALNERR\n");
		return EEPROM_ALNERR;
    }

	/* Verify that the block exists in the header */
	found = FALSE;
	for(i = 0; i < MAX_EEPROM_RECS && Eeprom_header.eep_recs[i].size; i++)
	{
		if(offset == Eeprom_header.eep_recs[i].offset && 
			size == Eeprom_header.eep_recs[i].size)
		{
			found = TRUE;
			break;
		}
	}
	if(found == FALSE)
	{
		if(offset != (u_short)EEP_HDR_ADDR || size != EEP_HDR_SIZE)
			return EEPROM_NOSUCH;
	}
	eep = (byte *)(EepromAddr + offset);
	/*
	 *	The first two bytes should be the marker
	 */
	if (*((ushort *)eep) != EEP_MARKER)
		{
		return EEPROM_BADMARKER;
		}
	cksump = (u_short *)(eep + size - 2);
	if((u_short)~cksum((u_short *)eep, (size - 2) >> 1) != *cksump)
		return EEPROM_CSERR;
	memcpy(bufadr,eep,size - 2);
	return EEPROM_AOK;
}

PutEeprom(offset,bufadr,size)
u_short	offset;
byte	*bufadr;
int	size;	
{
	int	found;
	u_short *cksump;
	int	i;

	if(((int)offset & 0x1f) || (size & 0x1f)) {
        printf("\nEEPROM_ALNERR\n");
		return EEPROM_ALNERR;
    }

	/* Verify that the block exists in the header */
	found = FALSE;
	for(i = 0; i < MAX_EEPROM_RECS && Eeprom_header.eep_recs[i].size; i++)
	{
		if(offset == Eeprom_header.eep_recs[i].offset)
		{
			/**
			if(size != Eeprom_header.eep_recs[i].size)
				return EEPROM_BADSIZE;
			**/
			if(size != Eeprom_header.eep_recs[i].size)
				{
                printf("\nEEPROM_BADSIZE");
				Eeprom_header.eep_recs[i].size = size;
				PutEeprom(EEP_HDR_ADDR, &Eeprom_header, EEP_HDR_SIZE);
				}
			found = TRUE;
			break;
		}
	}
	if(found == FALSE)
	{
		if (i >= MAX_EEPROM_RECS) {
            printf("\nEEPROM_NOSPACE");
			return EEPROM_NOSPACE;
        }
		/* A new block is being created */
		if((offset + size) > EEPROM_SIZE) {
            printf("\nEEPROM_BADSIZE 1");
			return EEPROM_BADSIZE;
        }
		Eeprom_header.eep_recs[i].size = size;
		Eeprom_header.eep_recs[i].offset = offset;
		PutEeprom(EEP_HDR_ADDR, &Eeprom_header, EEP_HDR_SIZE);
	}
	cksump = (u_short *)bufadr;
	*cksump = EEP_MARKER;	/* The First two byte should be the marker */
	cksump = (u_short *)(bufadr + size - 2);
	*cksump = ~cksum((u_short *)bufadr, (size - 2) >> 1);
	/*
	 *	Wait if the last PutEeprom is not done yet.
	 */
	while(EepromDirty)
		ReSchedule();
	EepromBufadr = bufadr;
	EepromOffset = (byte *)(EepromAddr + offset);
	Di();
	EepromDirty = GetEepBitmap(size);
	Ei();
	while(EepromDirty)
		{
		ReSchedule();
		}
	return EEPROM_AOK;
}

ProgEeprom()
{
	register u_int page;
	register int i;
	u_int	bitmap;
	byte	*eep_rec_ptr;
	byte	*ram_rec_ptr;

	if (EepromDirty == 0)
		return;		/* bitmap is 0,i.e.no pages to prog. */
	bitmap = EepromDirty;
	eep_rec_ptr = EepromOffset;
	ram_rec_ptr = EepromBufadr;
	for(page = 0; page != EEP_MAX_PAGES; page++, bitmap >>= 1)
		if (bitmap & 0x00000001)
			break;
	eep_rec_ptr += (page * EEP_PAGE_SIZE);
	ram_rec_ptr += (page * EEP_PAGE_SIZE);
	Di();
	EnableEeProm();
	*(u_int *)(eep_rec_ptr) = *(u_int *)(ram_rec_ptr);
	*(u_int *)(eep_rec_ptr + 4) = *(u_int *)(ram_rec_ptr + 4);
	*(u_int *)(eep_rec_ptr + 8) = *(u_int *)(ram_rec_ptr + 8);
	*(u_int *)(eep_rec_ptr + 12) = *(u_int *)(ram_rec_ptr + 12);
	*(u_int *)(eep_rec_ptr + 16) = *(u_int *)(ram_rec_ptr + 16);
	*(u_int *)(eep_rec_ptr + 20) = *(u_int *)(ram_rec_ptr + 20);
	*(u_int *)(eep_rec_ptr + 24) = *(u_int *)(ram_rec_ptr + 24);
	*(u_int *)(eep_rec_ptr + 28) = *(u_int *)(ram_rec_ptr + 28);
	EepromDirty &= ~(1 << page);
	DisableEeProm();
	Ei();
}

GetEepBitmap(size)
register word size;
{
	register word bitmap = 0xffffffff;
	register word page = size / EEP_PAGE_SIZE;

	bitmap <<= page;
	bitmap = ~bitmap;
	return bitmap;
}
