/*****************************************************************
 *
 *	Copyright (c) 1992 by Hughes LAN Systems
 *
 *	Program Name:	Token Ring Concentrator
 *
 *	Filename:	flash.c
 *
 *	Description:	This file contains the drivers and 
 *			support functions for the 28F008SA
 *			flash EPROM in Token Ring Concentrator
 *			application.
 *
 *	Revision:
 *	1-000		Aug-06-92		Szewei Ju
 *		- Added "volatile" type for all the pointer to
 *		flash EPROM location. so, consecutive write/read
 *		to the same location won't be optimized by the
 *		compiler.
 *	1-000		Aug-04-92		Szewei Ju
 *		- Removed "far" pointer declarations for i960
 *		environment.
 *		- Original, adapted from the Intel driver "8mbitdrv.c"
 *
 *******************************************************************
 */


/************************************************************************/
/*	Copyright Intel Corporation, 1992				*/
/*	Brian Dipert, Intel Corporation, May 7, 1992, Revision 2.1	*/
/*	The following drivers control the Command and Status Registers	*/
/*	of the 28F008SA Flash Memory to	drive byte write, block erase,	*/
/* 	Status Register read and clear and array read algorithms.	*/
/*	Sample Vpp an /PWD control blocks are also included, as are	*/
/*	example programs combining drivers into full algorithms		*/
/*	The functions listed below are included:			*/
/*		erasbgn(): Begins block erasure				*/
/*		erassusp(): Suspends block erase to allow reading data	*/
/*			from a block of the 28F008SA other than		*/
/*			that being erased				*/
/*		erasres(): Resumes block erase if suspended		*/
/*		end(): Polls the Write State Machine to determine	*/
/*			if block erase or byte write have completed	*/
/*		eraschk(): Executes full status check after block	*/
/*			erase completion				*/
/*		writebgn(): Begins byte write				*/
/*		writechk(): Executes full status check after byte	*/
/*			write completion				*/
/*		idread(): Reads and returns the manufacturer and	*/
/*			device IDs of the target 28F008SA		*/
/*		statrd(): Reads and returns the contents of the		*/
/*			Status Register					*/
/*		statclr(): Clears the Status Register			*/
/*		rdmode (): Puts the 28F008SA in Read Array mode		*/
/*		rdbyte (): Reads and returns a specified byte from	*/
/*			the target 28F008SA				*/
/*		vppup(): Enables high voltage Vpph			*/
/*		vppdown(): Disables Vpph				*/
/*		pwden(): Enables active low signal /PWD			*/
/*		pwddis(): Disables active low signal /PWD		*/
/*									*/
/*	Addresses are transferred to functions as pointers to		*/
/*	far bytes (ie long integers).  An alternate			*/
/*	approach is to create a global array the size of the 28F008SA	*/
/*	and located "over" the 28F008SA in the system memory map.	*/
/*	Accessing specific locations of the 28F008SA is then		*/
/*	accomplished by passing the chosen function an offset from	*/
/*	the array base versus a	specific address.  Different		*/
/*	microprocessor architectures will require different array	*/
/*	definitions; ie for the x86 architecture, define it as		*/
/*	"byte eightmeg[16][10000]" and pass each function TWO offsets	*/
/*	to access a specific location.  MCS-96 architectures are	*/
/*	limited to "byte eightmeg[10000]"; alternate approaches		*/
/*	such as using port pins for paging will be required to access	*/
/*	the full flash array						*/
/*									*/
/*	To create a far pointer, a function such as MK_FP() can be	*/
/*	used, given a segment and offset in the x86 architecture.	*/
/*	I use Turbo-C; see your compiler reference manual for		*/
/*	additional information.						*/
/************************************************************************/

/************************************************************************/
/*	Revision History: Rev 2.1					*/
/*									*/
/*	Changes From Revision 1.0 to Revision 2.0:			*/
/*		Added alternate 28F008SA device ID to routine idread()	*/
/*									*/
/*	Changes from 2.0 to 2.1:  Revised the Erase Suspend algorithm	*/
/*	to remove potential "infinate loop" caused by the WSM		*/
/*	going "ready" after the system reads the Status Register,	*/
/*	and before the system issues the Erase Suspend command		*/
/************************************************************************/

#include <types.h>

/********************************************
#include <memory.h>
********************************************/


/************************************************************************/
/*	Function: Main							*/
/*	Description: The following code (commented out) shows examples	*/
/*	of byte write and block erase algorithms that can be		*/
/*	modified to fit the specific application and hardware design	*/
/************************************************************************/


#ifdef yyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyy
main()
{
/*	byte *address;							*/
/*	byte data,status;						*/
/*									*/
/*	The following code gives an example of a possible byte write	*/
/*	algorithm. Note that Vpp does not need to be cycled between	*/
/*	byte writes when a string of byte writes occurs. Ramp Vpp to	*/
/*	12V before the first byte write and leave at 12V until after	*/
/*	completion of the last byte write.  Doing so minimizes Vpp ramp	*/
/*	up-down delay and maximizes byte write throughput		*/
/*	vppup();							*/
/*	"INSERT SOFTWARE DELAY FOR VPP RAMP IF REQUIRED	"		*/
/*	pwddis();							*/
/*	address	= 0XxxxxxL;						*/
/*	data	= 0Xyy;							*/
/*	if (writebgn(data,address) == 1)				*/
/*		"RECOVERY CODE-POWER NOT APPLIED (ID CHECK FAIL)"	*/
/*	else								*/
/*	{								*/
/*		while (end(&status) )					*/
/*			;						*/
/*		switch (writechk(status))				*/
/*		{							*/
/*			case 0:						*/
/*				break;					*/
/*			case 1:						*/
/*				"RECOVERY CODE-VPP LOW DETECT ERROR"	*/
/*				break;					*/
/*			case 2:						*/
/*				"RECOVERY CODE-BYTE WRITE ERROR"	*/
/*				break;					*/
/*		}							*/
/*		statclr();						*/
/*	}								*/
/*	vppdown();							*/
/*									*/
/*									*/
/*	The following code gives an example of a possible block erase	*/
/*	algorithm. Note that Vpp does not need to be cycled between	*/
/*	block erases when a string of block erases occurs. Ramp Vpp to	*/
/*	12V before the first block erase and leave at 12V until after	*/
/*	completion of the last block erase. Doing so minimizes Vpp ramp	*/
/*	up-down delay and maximizes block erase throughput		*/
/*	vppup();							*/
/*	"INSERT SOFTWARE DELAY FOR VPP RAMP IF REQUIRED	"		*/
/*	pwddis();							*/
/*	address	= 0XxxxxxL;						*/
/*	if (erasbgn(address) == 1)					*/
/*		"RECOVERY CODE-POWER NOT APPLIED (ID CHECK FAIL)"	*/
/*	else								*/
/*	{								*/
/*		while (end(&status) )					*/
/*			;						*/
/*		switch (eraschk(status))				*/
/*		{							*/
/*			case 0:						*/
/*				break;					*/
/*			case 1:						*/
/*				"RECOVERY CODE-VPP LOW DETECT ERROR"	*/
/*				break;					*/
/*			case 2:						*/
/*				"RECOVERY CODE-BLOCK ERASE ERROR"	*/
/*				break;					*/
/*			case 3:						*/
/*				"RECOVERY CODE-ERASE SEQUENCE ERROR"	*/
/*				break;					*/
/*		}							*/
/*		statclr();						*/
/*	}								*/
/*	vppdown();							*/
/*									*/
}
#endif /*yyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyy*/


/************************************************************************/
/*	Function: Erasgbn						*/
/*	Description: Begins erase of a block.				*/
/*	Inputs:	blckaddr: System address within the block to be erased	*/
/*	Outputs:	None						*/
/*	Returns:	0 = Block erase successfully initiated		*/
/*		1 = Block erase not initiated (ID check error)		*/
/*	Device Read Mode on Return: Status Register (ID if returns 1)	*/
/************************************************************************/

#define	ERASETUP	0X20		/* Erase Setup command		*/
#define	ERASCONF	0XD0		/* Erase Confirm command	*/

int erasbgn(blckaddr)
volatile byte *blckaddr;	/* blckaddr is an address within	*/
				/* the block to be erased		*/
{
#ifdef yyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyy
	byte mfgrid,deviceid;

	/* ID read error; device not powered up?	*/
	if (idread(&mfgrid,&deviceid)==1)
		return (1);
#endif /*yyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyy*/

	/* Write Erase Setup command to block address	*/
	*blckaddr = ERASETUP;

	/* Write Erase Confirm command to block address	*/
	*blckaddr = ERASCONF;

	return(0);
}


/************************************************************************/
/*	Function: Erassusp						*/
/*	Description: Suspends block erase to read from another block	*/
/*	Inputs:	None							*/
/*	Outputs: None							*/
/*	Returns: 0 = Block erase suspended				*/
/*		1 = Error; Write State Machine not busy			*/
/*		(block erase suspend not possible)			*/
/*	Device Read Mode on Return: Status Register			*/
/************************************************************************/

#define	RDYMASK		0X80		/* Mask to isolate the WSM	*/
					/* Statusbit of the Status	*/
					/* Register		 	*/
#define	WSMRDY		0X80		/* Status Register value after	*/
					/* masking, signifying that the	*/
					/* WSM is no longer busy	*/
#define	SUSPMASK	0X40		/* Mask to isolate the erase	*/
					/* suspend status bit of the	*/
					/* Status Register		*/
#define	ESUSPYES	0X40		/* Status Register value after	*/
					/* masking, signifying that	*/
					/* block erase has been		*/
					/* suspended			*/
#define	STATREAD	0X70		/* Read Status Register command	*/


#ifdef yyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyy
#define	SYSADDR		FEPROM		/* This constant can be		*/
					/* initialized to any address	*/
					/* within the memory map	*/
					/* of the target 28F008SA and	*/
					/* is alterable depending on	*/
					/* the system architecture	*/
#define	MFGRADDR	FEPROM		/* Address "0" for the target	*/
					/* 28F008SA...alterable		*/
					/* depending on the		*/
					/* system architecture		*/
#endif /*yyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyy*/


#define	SUSPCMD	0XB0		/* Erase Suspend command		*/


/*------------------------------------------------------------------------
* we make FEPROM, SYSADDR and MFGRADDR static variables,
* settable by application program via InitFlash()
* default values are suitable for token ring bridge archetecture
*------------------------------------------------------------------------*/
static word SYSADDR = 0xe0000000;
static word MFGRADDR = 0xe0000000;
static word FEPROM = 0xe0000000;


int erassusp()

{
	/* Pointer variable used to write commands to device	*/
	volatile byte *stataddr;

	stataddr = (byte *)SYSADDR;

	/* Write Erase Suspend command to the device	*/
	*stataddr = SUSPCMD;

	/* Write Read Status Register command..necessary in case */
	/* erase is already completed	*/
	*stataddr = STATREAD;

	/* Will remain in while loop until bit 7 of	*/
	/* the Status Register goes to 1, signifying	*/
	/* that the WSM is no longer busy		*/
	while ((*stataddr & RDYMASK) != WSMRDY)
		;

	if ((*stataddr & SUSPMASK) == ESUSPYES)
	/* Erase is suspended...return code "0"	*/
		return (0);

	/* Erase has already completed; suspend not possible. */
	/* Error code "1" */
	return(1);
}


/************************************************************************/
/*	Function: Erasres						*/
/*	Description: Resumes block erase previously suspended		*/
/*	Inputs:	None							*/
/*	Outputs:	None						*/
/*	Returns:	0 = Block erase resumed				*/
/*		1 = Error; Block erase not suspended when function	*/
/*		called							*/
/*	Device Read Mode on Return: Status Register			*/
/************************************************************************/

#define	RDYMASK		0X80		/* Mask to isolate the WSM	*/
					/* Status bit of the Status	*/
					/* Register			*/
#define	WSMRDY		0X80		/* Status Register value	*/
					/* after masking, signifying	*/
					/* that WSM is no longer busy	*/
#define	SUSPMASK	0X40		/* Mask to isolate the erase	*/
					/* suspend status bit of the	*/
					/* Status Register		*/
#define	ESUSPYES	0X40		/* Status Register value after	*/
					/* masking, signifying that	*/
					/* block erase has been		*/
					/* suspended			*/
#define	STATREAD	0X70		/* Read Status Register command	*/
#define	RESUMCMD	0XD0		/* Erase Resume command		*/

int erasres()
{
	volatile byte *stataddr;	/* Pointer variable used to	*/
					/* write commands to device	*/
	stataddr = (byte *)SYSADDR;

	/* Write Read Status Register command to 28F008SA */
	*stataddr = STATREAD;

	if ((*stataddr & SUSPMASK) != ESUSPYES)
	/* Block erase not suspended. Error code "1" */
		return (1);

	/* Write Erase Resume command to the device */
	*stataddr = RESUMCMD;

	/* Will remain in while loop until bit 6 */
	/* of the Status Register goes to 0, */
	/* signifying block erase resumption */
	while ((*stataddr & SUSPMASK) == ESUSPYES)
		;

	/* Will remain in while loop until bit 7 */
	/* of the Status Register goes to 0, */
	/* signifying that the WSM is once again busy */
	while ((*stataddr & RDYMASK) == WSMRDY)
		;

	return (0);
}

/************************************************************************/
/*	Function: End							*/
/*	Description: Checks to see if the WSM is busy			*/
/*		(is byte write/block erase completed?)			*/
/*	Inputs:	None							*/
/*	Outputs: statdata: Status Register data read from device	*/
/*	Returns: 0 = Byte Write/Block Erase completed			*/
/*		1 = Byte Write/Block Erase still in progress		*/
/*	Device Read Mode on Return: Status Register			*/
/************************************************************************/

#define	RDYMASK		0X80		/* Mask to isolate the WSM	*/
					/* Status bit of the Status	*/
					/* Register			*/
#define	WSMRDY		0X80		/* Status Register value after	*/
					/* masking, signifying that the	*/
					/* WSM is no longer busy	*/
#define	STATREAD	0X70		/* Read Status Register command	*/


int end_status(statdata)
byte *statdata;		/* Allows Status Register data to be passed	*/
			/* back to main program for further analysis	*/
{
	/* Pointer variable used to write commands to device	*/
	volatile byte *stataddr;

	stataddr = (byte *)SYSADDR;

	/* Write Read Status Register command to 28F008SA */
	*stataddr = STATREAD;

	/* Byte write/block erasure still in progress...code "1" */
	if (((*statdata = *stataddr) & RDYMASK) != WSMRDY)
		return (1);

	/* Byte write/block erase attempt completed...code "0"	*/
	return(0);
}


/************************************************************************/
/*	Function: Eraschk						*/
/*	Description: Completes full Status Register check for block	*/
/*	erase (proper command sequence, Vpp low detect, block erase	*/
/*	success). This routine assumes that block erase completion	*/
/*	has already been checked in function end(), and therefore	*/
/*	does not check the WSM Status bit of the Status Register	*/
/*	Inputs:	statdata: Status Register data read in function end	*/
/*	Outputs:	None						*/
/*	Returns:	0 = Block erase completed successfully		*/
/*		1 = Error; Vpp low detect				*/
/*		2 = Error; Block erase error				*/
/*		3 = Error; Improper command sequencing			*/
/*	Device Read Mode on Return: Same as when entered		*/
/************************************************************************/

#define	ESEQMASK	0X30		/* Mask to isolate the erase	*/
					/* and byte write status bits	*/
					/* of the Status Register	*/
#define	ESEQFAIL	0X30		/* Status Register value after	*/
					/* masking if block erase	*/
					/* command sequence error	*/
					/* has been detected		*/
#define	EERRMSK		0X20		/* Mask to isolate the erase	*/
					/* status bit of the Status	*/
					/* Register			*/
#define	ERASERR		0X20		/* Status Register value after	*/
					/* masking if block erase error	*/
					/* has been detected		*/
#define	VLOWMASK	0X08		/* Mask to isolate the Vpp	*/
					/* status bit of the Status	*/
					/* Register			*/
#define	VPPLOW		0X08		/* Status Register value after	*/
					/* masking if Vpp low has been	*/
					/* detected			*/

int eraschk(statdata)
byte statdata;				/* Status Register data that	*/
					/* has been already read from	*/
					/* the 28F008SA	in function	*/
					/* end()			*/
{
	if ((statdata & VLOWMASK) == VPPLOW)
	/* Vpp low detect error, return code "1" */
		return (1);

	if ((statdata & EERRMSK) == ERASERR)
	/* Block erase error detect, return code "2" */
		return (2);

	if ((statdata & ESEQMASK) == ESEQFAIL)
	/* Block erase command sequence error, return code "3" */
		return (3);

	return (0);	/* Block erase success, return code "0"	*/
}


/************************************************************************/
/*	Function: Writebgn						*/
/*	Description: Begins byte write sequence				*/
/*	Inputs:	wdata: Data to be written into the device		*/
/*		waddr: Target address to be written			*/
/*	Outputs:	None						*/
/*	Returns:	0 = Byte write successfully initiated		*/
/*		1 = Byte write not initiated (ID check error)		*/
/*	Device Read Mode on Return: Status Register (ID if returns 1)	*/
/************************************************************************/

#define	SETUPCMD	0X40		/* Byte Write Setup command	*/

int writebgn(wdata,waddr)
byte wdata;			/* Data to be written into the 28F008SA	*/
volatile byte *waddr;		/* waddr is the destination address for	*/
				/* the data to be written		*/
{
#ifdef yyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyy
	byte mfgrid,deviceid;

	if (idread(&mfgrid,&deviceid)==1)
	/* Device ID read error...powered up?	*/
		return (1);
#endif /*yyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyy*/

	/* Write Byte Write Setup command and destination address */
	*waddr 	= SETUPCMD;
	/* Write byte write data and destination address */
	*waddr 	= wdata;
	return (0);
}


/************************************************************************/
/*	Function: Writechk						*/
/*	Description: Completes full Status Register check for byte	*/
/*	write (Vpp low detect, byte write success). 			*/
/*		This routine assumes that byte write completion has	*/
/*		already been checked in function end() and therefore	*/
/*		does not check the WSM Status bit of Status Register	*/
/*	Inputs:	statdata: Status Register data read in function end()	*/
/*	Outputs:	None						*/
/*	Returns:	0 = Byte write completed successfully		*/
/*		1 = Error; Vpp low detect				*/
/*		2 = Error; Byte write error				*/
/*	Device Read Mode on Return: Status Register			*/
/************************************************************************/

#define	WERRMSK		0X10		/* Mask to isolate the byte	*/
					/* write error bit of the	*/
					/* Status Register		*/
#define	WRITERR		0X10		/* Status Register value after	*/
					/* masking if byte write error	*/
					/* has been detected 		*/
#define	VLOWMASK	0X08		/* Mask to isolate the Vpp	*/
					/* status bit of the Status	*/
					/* Register			*/
#define	VPPLOW		0X08		/* Status Register value after	*/
					/* masking if Vpp low has been	*/
					/* detected			*/

int writechk(statdata)
byte statdata;			/* Status Register data that has been	*/
				/* already read from the 28F008SA	*/
				/* in function end()			*/
{
	if ((statdata & VLOWMASK) == VPPLOW)
	/* Vpp low detect error, return code "1" */
		return (1);

	if ((statdata & WERRMSK) == WRITERR)
	/* Byte write error detect, return code "2" */
		return (2);

	/* Byte/string write success, return code "0" */
	return (0);
}


/************************************************************************/
/*	Function: Idread						*/
/*	Description: Reads the manufacturer and device IDs from the	*/
/*		target 28F008SA						*/
/*	Inputs:	None							*/
/*	Outputs:	mfgrid: Returned manufacturer ID		*/
/*		deviceid: Returned device ID				*/
/*	Returns:	0 = ID read correct				*/
/*		1 = Wrong or no ID					*/
/*	Device Read Mode on Return: inteligent Identifier		*/
/************************************************************************/
#define	DEVICADD	1		/* Address "1" for the target	*/
					/* 28F008SA...alterable		*/
					/* depending on the		*/
					/* system architecture		*/
#define	IDRDCOMM	0X90		/* Inteligent Identifier command*/
#define INTELID		0X89		/* Manufacturer ID for Intel	*/
					/* devices			*/
#define DVCID		0X0A2		/* Device IDs for 28F008SA	*/
#define	DVCID2		0X0A1

int idread(mfgrid,deviceid)
byte *mfgrid;				/* The manufacturer ID read by	*/
					/* this function, to be		*/
					/* transferred back to		*/
					/* the calling program		*/
byte *deviceid;				/* The device ID read by this	*/
					/* function, to be transferred	*/
					/* back to thecalling function	*/
{
	/* Pointer address variable used to read IDs */
	volatile byte *tempaddr;

	tempaddr = (byte *)MFGRADDR;

	/* Write inteligent Identifier command to an */
	/* address within the 28F008SA */
	/* memory map (in this case 00 hex) */
	*tempaddr = IDRDCOMM;

	/* Read mfgr ID, tempaddr still points at address "0" */
	*mfgrid	= *tempaddr++;

	/* Read device ID */
	*deviceid = *tempaddr;

	if ((*mfgrid != INTELID) ||
	((*deviceid != DVCID) && (*deviceid != DVCID2)))
		return (1); /* ID read error; device powered up? */

	return (0);
}


/************************************************************************/
/*	Function: Statrd						*/
/*	Description: Returns contents of the target 28F008SA		*/
/*		Status Register						*/
/*	Inputs:	None							*/
/*	Outputs:	statdata: Returned Status Register data		*/
/*	Returns:	Nothing						*/
/*	Device Read Mode on Return: Status Register			*/
/************************************************************************/

#define	STATREAD	0X70		/* Read Status Register command				*/

int statrd(statdata)
byte *statdata;				/* Allows Status Register data	*/
					/* to be passed back to the	*/
					/* calling program		*/
					/* for further analysis		*/
{
	/* Pointer variable used to write commands to device */
	volatile byte *stataddr;

	stataddr = (byte *)SYSADDR;
	/* Write Read Status Register command to 28F008SA */
	*stataddr = STATREAD;
	*statdata = *stataddr;
	return;
}


/************************************************************************/
/*	Function: Statclr						*/
/*	Description: Clears the 28F008SA Status Register		*/
/*	Inputs:	None							*/
/*	Outputs:	None						*/
/*	Returns:	Nothing						*/
/*	Device Read Mode on Return: Array				*/
/************************************************************************/

#define	STATCLER	0X50	/* Clear Status Register command */

int statclr()
{
	/* Pointer variable used to write commands to device */
	byte *stataddr;

	stataddr = (byte *)SYSADDR;
	/* Write Clear Status Register command to 28F008SA */
	*stataddr = STATCLER;
	return;
}


/************************************************************************/
/*	Function: Rdmode						*/
/*	Description: Puts the target 28F008SA in Read Array Mode.	*/
/*	This function might be used, for example, to prepare		*/
/*		the system for return to code execution out of the	*/
/*		Flash memory after byte write or block erase algorithms	*/
/*		have been executed off-chip				*/
/*	Inputs:	None							*/
/*	Outputs:	None						*/
/*	Returns:	Nothing						*/
/*	Device Read Mode on Return: Array				*/
/************************************************************************/

#define	RDARRAY	0XFF		/* Read Array command	*/

int rdmode()
{
	/* Pointer variable used to write commands to the device */
	byte *tempaddr;

	tempaddr = (byte *)SYSADDR;
	/* Write Read Array command to 28F008SA	*/
	*tempaddr = RDARRAY;
	return;
}


#ifdef yyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyy
/*--------------------------------------------------------------------------
 *	Delay 10 ms
 */
FlashDelay10Ms()
{
	int	ticks = RealTimeTicks();

	while ((RealTimeTicks() - ticks) <= 2)
		ReSchedule();
}

/*** these archetecture dependent routines are provided by application ***/
/*--------------------------------------------------------------------------
 *	Turn on the Vpp for programing flash EPROM
 */
void VppUp()
{
	byte	*vppctrl;

	vppctrl = (byte *)0x80000036;
	*vppctrl = 1;			/* raise the VPP to flash EPROM */
	FlashDelay10Ms();		/* Vpp ramp delay */
}

/*--------------------------------------------------------------------------
 *	Turn off the Vpp, stop programming flash EPROM
 */
void VppDown()
{
	byte	*vppctrl;

	vppctrl = (byte *)0x80000036;
	*vppctrl = 0;			/* lower the VPP to flash EPROM */
	FlashDelay10Ms();
}
#endif /*yyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyy*/


/*--------------------------------------------------------------------------
* Write one byte data into the flash memory and return the result
* of the operation
* Return: 0 -- OK.
* other -- error.
*--------------------------------------------------------------------------*/
int WriteFlash( byte *address, byte data)
{
	byte	flashStatus;

	writebgn( data, address);	
	while ( end_status( &flashStatus) ) /* wait for write to complete */
			;

	/* check and return the result of the write */
	return( writechk( flashStatus));
}


/*--------------------------------------------------------------------------
* Erase a 64K block of the 28F008SA flash EPROM, and check the erased content
* "blockaddr" must be at the 64K boundary.
* return 0 if ok
* return 1 if failed after nRetry
* return 2 if read verify error
*--------------------------------------------------------------------------*/
int EraseFlashBlock( byte *blockaddr, int nRetry)
{
	byte	flashStatus, *pAddr;
	int i, j;

	for ( i=0; i<nRetry; i++)
	{
		statclr();	/* clear the status register */
		VppUp();	/* raise the Vpp to flash EPROM */
		/* setup and confirm the erase command */
		erasbgn( blockaddr);

		/* wait for erasure to complete */
		while ( end_status( &flashStatus) )
			;

		VppDown();
		/* check the result of the block erasure */
		if ( eraschk( flashStatus) )
			continue;

		rdmode();	/* set the flash EPROM in read array mode */
		for ( j=0, pAddr=blockaddr; j < 0x10000; j++, pAddr++)
			if ( *pAddr != 0xFF )	/* check the erased content */
				return 2;	/* erase readback error */
		return 0;
	}
	return 1;	/* error during erase */
}


/*-----------------------------------------------------------------------
* Write a block of data into the flash memory
* Return 1 -- OK.
* Return 0 -- error.
*-----------------------------------------------------------------------*/
int
ProgramFlash(void *feprom, unsigned short *DataPtr, int size)
{
	byte *pBlkAddr, *src, *dst;
	int block, retcode, nByte;
	int start_block;
	int end_block;

	if ( size < 0)
		return 1;
	printf("program flash: %lx -> %lx (%lx)\n", DataPtr, feprom, size);

	/* Erase the flash EPROM */
	printf("Erasing flash eprom ");
	start_block = ((ulong)feprom - (ulong)FEPROM) >> 16;
	end_block = start_block + (size >> 16) + 1;
	printf("start block = %d, end block = %d\n", start_block, end_block);
	for (block = start_block; block <= end_block; block++)
	{
		pBlkAddr = (byte *) FEPROM + (block << 16);
		if ( EraseFlashBlock(pBlkAddr, 1) )
			return 1;
		printf("-");
	}
	printf("\n");

	/* Burn the flash EPROM */
	printf("Burning flash eprom: ");
	src = (byte *)DataPtr;
	dst = (byte *)feprom;
	nByte = size;
	printf("%lx -> %lx (%lx)\n", src, dst, nByte);
	statclr();	/* clear the status register */
	VppUp();
	while ( nByte--)
	{
		if (retcode=WriteFlash( dst, *src))
		{
			VppDown();
			return 1;
		}
		src++;
		dst++;
		if ((nByte & 0xffff) == 0)
			printf("+");
	}
	printf("+\n");
	VppDown();

	/* read back and compare */
	printf("Comparing flash eprom with memory: ");
	src = (byte *)DataPtr;
	dst = (byte *)feprom;
	nByte = size;
	printf("%lx, %lx (%lx)\n", src, dst, nByte);
	rdmode();		/* set the flash EPROM in read array mode */
	while (nByte--)
	{
		if (*src != *dst)
		{
			printf("error at %lx, %lx\n", src, dst);
/*****			FlashSLed(3, 0);		********/
			return 1;
		}
		src++;
		dst++;
		if ((nByte & 0xffff) == 0)
			printf("*");
	}
	printf("*\n");
	return 0;
}


/*-----------------------------------------------------------------------
*	Get and check the manufacture ID and device Id of the flash EPROM
* Return: 0 -- OK.
*	   other -- error.
*-----------------------------------------------------------------------*/
int CheckFlash( void *feprom, int size)
{
	byte mftrid=0, devid=0;
	int retcode;

	retcode = idread( &mftrid, &devid); 
	printf( "Flash EPROM manufacture ID = %X, device ID = %X\n",
			mftrid, devid);
	if ( retcode )
	{
		printf( "\tError: Unknown Flash EPROM type.");
		return 1;
	}
	printf( "\nFlash EPROM type OK.\n");
	return 0;
}


/*--------------------------------------------------------------------
 *	Test a block of flash EPROM  by address
*--------------------------------------------------------------------*/
int TestFlashBlockAddress( byte *pBlkAddr)
{
	int i;
	byte *pAddr, pattern;
/*
 * erase the flash EPROM  block
 */
	printf( "\nErase flash EPROM block at %x ... ", pBlkAddr);
	if ( EraseFlashBlock( pBlkAddr, 1) )
	{
		printf( "error, test aborted.");
		return 1;
	}
	printf( "OK");
/*
 * Do address test
 */
	printf( "\nFlash EPROM address test ..");
	statclr();	/* clear the status register */
	VppUp();
	for ( i=0, pAddr = pBlkAddr; i < 0x10000; i++, pAddr++)
	{
		pattern = (byte)pAddr;
		if ( WriteFlash( pAddr, pattern) )
		{
			printf( "\nError: Addr test, write flash EPROM at %x.",
					pAddr);
			VppDown();
			return 1;
		}
		if ( !(i%0x1000) )
			printf( ".");
	}
	VppDown();
	rdmode();		/* set the flash EPROM in read array mode */
	for ( i=0, pAddr = pBlkAddr; i < 0x10000; i++, pAddr++)
	{
		pattern = (byte)pAddr;
		if ( *pAddr != pattern )
		{
			printf( "\nError: flash EPROM address test at %x.",
					pAddr);
			return 1;
		}
		if ( !(i%0x1000) )
			printf( ".");
	}
	printf( "OK");

	return 0;
}


/*-----------------------------------------------------------------------
*	Test a block of flash EPROM use a suppied data pattern 
*-----------------------------------------------------------------------*/
int TestFlashBlockData( byte *pBlkAddr, byte pattern)
{
	int i;
	byte *pAddr;
/*
 * erase the flash EPROM  block
 */
	printf( "\nErase flash EPROM block at %x ... ", pBlkAddr);
	if ( EraseFlashBlock( pBlkAddr, 1) )
	{
		printf( "error, test aborted.");
		return 1;
	}
	printf( "OK");
/*
 * Do data test use the pattern 
 */
	printf( "\nFlash EPROM data test (%x)", pattern);
	statclr();	/* clear the status register */
	VppUp();
	for ( i=0, pAddr = pBlkAddr; i < 0x10000; i++, pAddr++)
	{
		if ( WriteFlash( pAddr, pattern) )
		{
			printf( "\nError: Writing flash EPROM at %x.", pAddr);
			VppDown();
			return 1;
		}
		if ( !(i%0x1000) )
			printf( ".");
	}
	VppDown();
	rdmode();		/* set the flash EPROM in read array mode */
	for ( i=0, pAddr = pBlkAddr; i < 0x10000; i++, pAddr++)
	{
		if ( *pAddr != pattern )
		{
			printf( "\nError: Reading flash EPROMat %x.", pAddr);
			return 1;
		}
		if ( !(i%0x1000) )
			printf( ".");
	}
	printf( "OK");

	return 0;
}


/*-----------------------------------------------------------------------
*	Test to program the flash EPROM, first to erase to FF,
*	then do data/address test.
*-----------------------------------------------------------------------*/
int TestProgramFlash( void *feprom)
{
	byte *pBlkAddr;
	int block;

	printf("feprom = %X\n",feprom);
	/* test block by block */
	for ( block=1, pBlkAddr = (byte *)feprom; block <= 16;
	block++, pBlkAddr += 0x10000)
	{
		if ( TestFlashBlockData( pBlkAddr, 0x5a) )
			{
			printf("\n");
			return 1;
			}

		if ( TestFlashBlockData( pBlkAddr, 0xa5) )
			{
			printf("\n");
			return 1;
			}

		if ( TestFlashBlockAddress( pBlkAddr) )
			{
			printf("\n");
			return 1;
			}
	}
	printf("\n");
	return 0;
}


/*------------------------------------------------------------------
* init static locations SYSADDR and MFGRADDR with "start"
*------------------------------------------------------------------*/
InitFlash(start)
word start;
{
	SYSADDR = start;
	MFGRADDR = start;
	FEPROM = start;
}
