/*==========================================================================
  ABLIT.C

  Example code to perform a screen to screen blit using the small VGA 32K
  apertures. A multi-colored filled rectangle is drawn on screen as the
  source to be blited. This source is blited by copying the pixel data
  to another screen region using the VGA 32K apertures.

  This example requires that the VGA aperture is NOT disabled.

  Copyright (c) 1994-1995 ATI Technologies Inc. All rights reserved
 =========================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <dos.h>
#include "..\util\atim64.h"
#include "..\util\sample.h"

#define INSTANCE 0

/* --------------------------------------------------------------------------
  GETPAGEPTR - calculate 32K read/write pointer from a given memory address.
-------------------------------------------------------------------------- */
unsigned long getpageptr (unsigned long addr)
{
    unsigned long pageptr;

    pageptr = addr / 0x8000;

    return (pageptr);
}

/* --------------------------------------------------------------------------
  GETADDR - calculate the memory address from a given set of coordinates.

  A byte address is returned.
-------------------------------------------------------------------------- */
unsigned long getaddr (int x, int y)
{
    unsigned long addr, xadd;

    // calculate byte address from given coordinates (x, y)
    xadd = (unsigned long) x;
    addr = (unsigned long) y;
    addr = (unsigned long) (addr * modeinfo.pitch);
    if (modeinfo.bpp == 4)
    {
        addr = addr / 2;
        xadd = xadd / 2;
    }
    else
    {
        addr = (unsigned long) (addr * (modeinfo.bpp / 8));
        xadd = (unsigned long) (xadd * (modeinfo.bpp / 8));

    }
    addr = addr + xadd;

    return (addr);
}

/* --------------------------------------------------------------------------
  GETOFFSET - calculate offset into 32K VGA aperture from the physical
              address and page pointer.
-------------------------------------------------------------------------- */
unsigned long getoffset (unsigned long phyaddr, unsigned long pageptr)
{
    unsigned long ptraddr, offset;

    ptraddr = pageptr * 0x8000;
    offset = phyaddr - ptraddr;

    return (offset);
}

/* --------------------------------------------------------------------------
  APPBLIT - perform a screen to screen blit using small 32K VGA apertures.

  This routine uses the lower read 32K VGA aperture as the source buffer to
  read a line of screen memory. This is transferred to the lower write 32K
  VGA aperture which represents the destination buffer. The upper 32K VGA
  aperture is used large block transfers or for overflows from the lower
  32K aperture.

  It is assumed that the VGA aperture is enabled.
-------------------------------------------------------------------------- */
void appblit (int x1, int y1, int x2, int y2, int width, int height)
{
    unsigned long readaddr, writeaddr;
    unsigned long readpage, writepage;
    unsigned long readoffset, writeoffset;
    int ySrc, yDst, index;
    int linesize;

    // calculate linesize in bytes
    if (modeinfo.bpp == 4)
    {
        linesize = width / 2;
    }
    else
    {
        linesize = width * (modeinfo.bpp / 8);
    }

    // transfer source to destination
    for (ySrc = y1, yDst = y2; ySrc < (y1 + height); ySrc++, yDst++)
    {
        // set read pointer to next source line address
        readaddr = getaddr (x1, ySrc);
        readpage = getpageptr (readaddr);
        readoffset = getoffset (readaddr, readpage);

        // upper 32k VGA aperture pointer is used for overflow
        regw (MEM_VGA_RP_SEL, ((readpage + 1) << 16) | readpage);

        // set write pointer to next destination line address
        writeaddr = getaddr (x2, yDst);
        writepage = getpageptr (writeaddr);
        writeoffset = getoffset (writeaddr, writepage);

        // upper 32k VGA aperture pointer is used for overflow
        regw (MEM_VGA_WP_SEL, ((writepage + 1) << 16) | writepage);

        // Reduce VGA aperture to 64KB addressable A000:0000-A000:FFFF as
        // some mach64CTs don't access the framebuffer properly othewise.
        outpw (0x3CE, 0x0506);      // graf06 = 05  (64K, graphics mode)

        // perform screen memory copy of one line
        for (index = 0; index < linesize; index++)
        {
            *((unsigned char far *) (LOW_APERTURE_BASE + writeoffset + index)) =
            *((unsigned char far *) (LOW_APERTURE_BASE + readoffset + index));
        }

        // Restore VGA aperture to 128KB addressable A000:0000-B000:FFFF in
        // order to access the memory mapped registers starting at B000:FC00
        outpw (0x3CE, 0x0106);      // graf06 = 01  (128K, graphics mode)
    }
}


/* Main C program */

int main (int argc, char *argv[])
{
    // check if Mach64 adapter is installed
    if (detect_mach64 (INSTANCE) != YES_MACH64)
    {
        printf ("mach64 based adapter was not found.\n");
        return (1);
    }

    // fill global query structure by calling Mach 64 ROM
    if (query_hardware () != NO_ERROR)
    {
        printf ("Failed ROM call to query mach64 hardware.\n");
        return (1);
    }

    // check if Mach 64 VGA controller is enabled
    if (querydata.vga_type != VGA_ENABLE)
    {
        printf ("This sample code example requires an enabled mach64 VGA "
                "controller.\n");
        return (1);
    }

    // The VGA controller is normally set in planar mode. Data transfer
    // through the VGA aperture (low and high 32K pages) requires that the
    // VGA controller be set in a packed pixel mode where the pixel data
    // is arranged contigiously.

    // Process the command line arguments to override default resolution
    // and color depth settings.
    process_command_line (argc, argv);

    set_packed_pixel ();

    // set an accelerator mode
    if (open_mode (gmode_res, PITCH_XRES, gclr_depth) != NO_ERROR)
    {
        close_mode ();
        printf ("Error in setting display mode.\n");
        return (1);
    }

    // Initialize standard engine context
    init_engine ();
    clear_screen (0, 0, modeinfo.xres, modeinfo.yres);

    // Draw some rectangles to act as a blit source
    set_fg_color (get_color_code (YELLOW));
    draw_rectangle (0, 0, modeinfo.xres / 2, modeinfo.yres / 2);

    set_fg_color (get_color_code (LIGHTRED));
    draw_rectangle ((modeinfo.xres / 32),
                    (modeinfo.yres / 24),
                    (modeinfo.xres / 2) - (2 * (modeinfo.xres / 32)),
                    (modeinfo.yres / 2) - (2 * (modeinfo.yres / 24)));

    set_fg_color (get_color_code (LIGHTGREEN));
    draw_rectangle ((2 * (modeinfo.xres / 32)),
                    (2 * (modeinfo.yres / 24)),
                    (modeinfo.xres / 2) - (4 * (modeinfo.xres / 32)),
                    (modeinfo.yres / 2) - (4 * (modeinfo.yres / 24)));

    set_fg_color (get_color_code (LIGHTBLUE));
    draw_rectangle ((3 * (modeinfo.xres / 32)),
                    (3 * (modeinfo.yres / 24)),
                    (modeinfo.xres / 2) - (6 * (modeinfo.xres / 32)),
                    (modeinfo.yres / 2) - (6 * (modeinfo.yres / 24)));

    // Draw a DARKGRAY rectangle to show where the destination blit will be
    // drawn
    set_fg_color (get_color_code (DARKGRAY));
    draw_rectangle (modeinfo.xres / 2, modeinfo.yres / 2,
                    modeinfo.xres / 2, modeinfo.yres / 2);

    // wait for a carriage return
    getch ();

    // perform blit using small 32K VGA apertures
    appblit (0, 0, modeinfo.xres / 2, modeinfo.yres / 2,
             modeinfo.xres / 2, modeinfo.yres / 2);

    // wait for a carriage return
    getch ();

    // disable accelerator mode and switch back to VGA text mode
    close_mode ();

    switch (gmode_res)
    {
        case MODE_640x480:
            printf ("mode = 640 x 480\n");
            break;

        case MODE_800x600:
            printf ("mode = 800 x 600\n");
            break;

        case MODE_1024x768:
            printf ("mode = 1024 x 768\n");
            break;

        case MODE_1280x1024:
            printf ("mode = 1280 x 1024\n");
            break;

        default:
            printf ("unknown mode\n");
            break;
    } // end switch mode_res


    switch (gclr_depth)
    {
        case COLOR_DEPTH_4:
            printf ("color depth = 4\n");
            break;

        case COLOR_DEPTH_8:
            printf ("color depth = 8\n");
            break;

        case COLOR_DEPTH_15:
            printf ("color depth = 15\n");
            break;

        case COLOR_DEPTH_16:
            printf ("color depth = 16\n");
            break;

        case COLOR_DEPTH_24:
            printf ("color depth = 24\n");
            break;

        case COLOR_DEPTH_32:
            printf ("color depth = 32\n");
            break;

        default:
            printf ("unknown color depth\n");
            break;
    } // end switch clr_depth

    return (0);
}

