; ===========================================================================
; VISR.ASM
;
; Interrupt service routine for Mach 64 vertical line interrupt.
;
; Compiling:
;   masm /Ml /D<memory model> romcalls.asm;
;       <memory model> = mem_S for SMALL model,
;                        mem_M for MEDIUM model,
;                        mem_L for LARGE model
;
; Copyright (c) 1994-1995 ATI Technologies Inc. All rights reserved
; ===========================================================================

include ..\util\atim64.inc

IFDEF mem_S
PARM        equ     4   ; passed parameters start at bp+4 for small model
ELSE
PARM        equ     6   ; passed parameters start at bp+6 for other models
ENDIF

IFDEF mem_S
.MODEL  SMALL, C
ELSEIFDEF mem_M
.MODEL  MEDIUM, C
ELSE
.MODEL  LARGE, C
ENDIF

.DATA

.CODE
.386

OldIntVector    dw  ?
                dw  ?
IRQnum          db  ?
IRQindex        db  ?
IRQmask1        db  ?
IRQmask2        db  ?
Count           db  0
Iwidth          dw  0
Height          dw  0
SrcX            dw  0
SrcY            dw  0
SaveX           dw  0
SaveY           dw  0
Ydraw           dw  0
Step            dw  0
Xstart          dw  0
Xpos            dw  0
Image           dw  0

IFDEF mem_S
extrn       ior8:NEAR
extrn       ior16:NEAR
extrn       ior32:NEAR
extrn       iow8:NEAR
extrn       iow16:NEAR
extrn       iow32:NEAR
ELSE
extrn       ior8:FAR
extrn       ior16:FAR
extrn       ior32:FAR
extrn       iow8:FAR
extrn       iow16:FAR
extrn       iow32:FAR
ENDIF

; Macro for 'call' model handling
Mcall       macro   routine
IFDEF mem_S
            call    NEAR PTR routine
ELSE
            call    FAR PTR routine
ENDIF
            endm

; macros to handle IO operations
Min         macro   inp_routine
            push    dx
            Mcall   inp_routine
            add     sp, 2
            endm

Mout        macro   outp_routine
            push    ax
            push    dx
            Mcall   outp_routine
            add     sp, 4
            endm

; ---------------------------------------------------------------------------
; DO_BLIT
;
; INPUT:  WORD x1,
;         WORD y1,
;         WORD x2,
;         WORD y2,
;         WORD width,
;         WORD height
;
; OUTPUT: none
;
; Assumes VGA aperture is enabled
; ---------------------------------------------------------------------------
            public  do_blit

do_blit     proc    far

            ; create frame pointer
            push    bp
            mov     bp, sp

            ; Save used registers
            push    es

            ; Wait for fifo in FIFO_STAT (using VGA aperture)
            mov     ax, 0B000h
            mov     es, ax

waitfifo:
            mov     ax, es:[0FC00h+FIFO_STAT]
            or      ax, ax
            jnz     waitfifo

            ; Load up blit registers
            xor     eax, eax

            ; SRC_X = x1
            mov     ax, WORD PTR [bp+PARM]
            mov     DWORD PTR es:[0FC00h+SRC_X], eax

            ; SRC_Y = y1
            mov     ax, WORD PTR [bp+PARM+2]
            mov     DWORD PTR es:[0FC00h+SRC_Y], eax

            ; SRC_HEIGHT1 = height
            mov     ax, WORD PTR [bp+PARM+10]
            mov     DWORD PTR es:[0FC00h+SRC_HEIGHT1], eax

            ; SRC_WIDTH1 = width
            mov     ax, WORD PTR [bp+PARM+8]
            mov     DWORD PTR es:[0FC00h+SRC_WIDTH1], eax

            ; DST_X = x2
            mov     ax, WORD PTR [bp+PARM+4]
            mov     DWORD PTR es:[0FC00h+DST_X], eax

            ; DST_Y = y2
            mov     ax, WORD PTR [bp+PARM+6]
            mov     DWORD PTR es:[0FC00h+DST_Y], eax

            ; DST_HEIGHT = height
            mov     ax, WORD PTR [bp+PARM+10]
            mov     DWORD PTR es:[0FC00h+DST_HEIGHT], eax

            ; DST_WIDTH = width
            mov     ax, WORD PTR [bp+PARM+8]
            mov     DWORD PTR es:[0FC00h+DST_WIDTH], eax

            ; restore saved registers
            pop     es

            ; remove frame pointer
            mov     sp, bp
            pop     bp

            ret

do_blit     endp

; ---------------------------------------------------------------------------
; VlineISR - Handler for VLINE interrupt
;
; The Mach64 can have one of four IRQs set: 2, 3, 5, 10. It is assumed that
; the CRTC_VLINE_INT_EN bit is set in the CRTC_INT_CNTL register.
; ---------------------------------------------------------------------------
            public  VlineISR

VlineISR    proc    far

            cli

            push     ax
            push     dx

            ; Check if this interrupt trigger belongs to the VLINE interrupt
            mov      dx, ioCRTC_INT_CNTL
            Min      ior16
            and      ax, 18h
            jnz      acknowledge
            jmp      skip

acknowledge:
            ; Increment count
            inc      Count

            ; Ack 8259 interrupt controllers according to the IRQ number
            cmp      IRQnum, 0Ah
            je       ack_2nd_8259

            ; ack master 8259 - required for IRQ 2, 3, 5
            mov      dx, 20h
            mov      al, 20h
            out      dx, al

            cmp      IRQnum, 2
            jne      ack_vline

ack_2nd_8259:
            ; ack cascaded 8259 - required for IRQ 2, 10
            mov      dx, 0A0h
            mov      al, 20h
            out      dx, al

ack_vline:
            ; ack vline interrupt bit on Mach64
            mov      dx, ioCRTC_INT_CNTL
            Min      ior16
            or       ax, 10h
            Mout     iow16

            ; ---- restore (if needed), save, and update frame ----

            ; don't restore unless x position is more the zero
            cmp      Xpos, 0
            je       no_restore

            ; restore from last frame
            mov      ax, Height
            push     ax
            mov      ax, Iwidth
            push     ax
            mov      ax, Ydraw
            push     ax
            mov      ax, Xpos
            sub      ax, Step
            push     ax
            mov      ax, SaveY
            push     ax
            mov      ax, SaveX
            push     ax
            call     FAR PTR do_blit
            pop      ax
            pop      ax
            pop      ax
            pop      ax
            pop      ax
            pop      ax

no_restore:
            ; save frame
            mov      ax, Height
            push     ax
            mov      ax, Iwidth
            push     ax
            mov      ax, SaveY
            push     ax
            mov      ax, SaveX
            push     ax
            mov      ax, Ydraw
            push     ax
            mov      ax, Xpos
            push     ax
            call     FAR PTR do_blit
            pop      ax
            pop      ax
            pop      ax
            pop      ax
            pop      ax
            pop      ax

            ; update current frame
            mov      ax, Height
            push     ax
            mov      ax, Iwidth
            push     ax
            mov      ax, Ydraw
            push     ax
            mov      ax, Xpos
            push     ax
            mov      ax, SrcY
            push     ax
            mov      ax, SrcX
            push     ax
            call     FAR PTR do_blit
            pop      ax
            pop      ax
            pop      ax
            pop      ax
            pop      ax
            pop      ax

skip:
            pop      dx
            pop      ax

            pushf
            call     DWORD PTR OldIntVector

            sti

            iret

VlineISR    endp

; ---------------------------------------------------------------------------
; GETCOUNT
;
; Retrieve the count value controlled by the ISR.
;
; Inputs:  none
;
; Outputs: WORD count
; ---------------------------------------------------------------------------
            public  getcount

IFDEF mem_S
getcount    proc    near
ELSE
getcount    proc    far
ENDIF
            ; get count
            mov     al, Count
            xor     ah, ah

            ret

getcount    endp

; ---------------------------------------------------------------------------
; SETBLITINFO
;
; Set blit dimension information for ISR.
;
; Inputs:  WORD image width,
;          WORD image height,
;          WORD source x,
;          WORD source y,
;          WORD save x,
;          WORD save y,
;          WORD y draw,
;          WORD step
;
; Outputs: none
; ---------------------------------------------------------------------------
            public  setblitinfo

IFDEF mem_S
setblitinfo proc    near
ELSE
setblitinfo proc    far
ENDIF
            ; create frame pointer
            push    bp
            mov     bp, sp

            ; Save used registers
            push    ax

            ; get image width
            mov     ax, WORD PTR [bp+PARM]
            mov     Iwidth, ax

            ; get image height
            mov     ax, WORD PTR [bp+PARM+2]
            mov     Height, ax

            ; get source x
            mov     ax, WORD PTR [bp+PARM+4]
            mov     SrcX, ax

            ; get source y
            mov     ax, WORD PTR [bp+PARM+6]
            mov     SrcY, ax

            ; get save x
            mov     ax, WORD PTR [bp+PARM+8]
            mov     SaveX, ax

            ; get save y
            mov     ax, WORD PTR [bp+PARM+10]
            mov     SaveY, ax

            ; get y draw
            mov     ax, WORD PTR [bp+PARM+12]
            mov     Ydraw, ax

            ; get step
            mov     ax, WORD PTR [bp+PARM+14]
            mov     Step, ax

            ; restore saved registers
            pop     ax

            ; remove frame pointer
            mov     sp, bp
            pop     bp

            ret

setblitinfo endp

; ---------------------------------------------------------------------------
; SETXSTART
;
; Set image starting position for the ISR.
;
; Inputs:  WORD x image position
;
; Outputs: none
; ---------------------------------------------------------------------------
            public  setxstart

IFDEF mem_S
setxstart   proc    near
ELSE
setxstart   proc    far
ENDIF
            ; create frame pointer
            push    bp
            mov     bp, sp

            ; Save used registers
            push    ax

            ; get image starting position (x)
            mov     ax, WORD PTR [bp+PARM]
            mov     Xstart, ax

            ; restore saved registers
            pop     ax

            ; remove frame pointer
            mov     sp, bp
            pop     bp

            ret

setxstart   endp

; ---------------------------------------------------------------------------
; SETXPOS
;
; Set current image position for ISR.
;
; Inputs:  WORD x image position
;
; Outputs: none
; ---------------------------------------------------------------------------
            public  setxpos

IFDEF mem_S
setxpos     proc    near
ELSE
setxpos     proc    far
ENDIF
            ; create frame pointer
            push    bp
            mov     bp, sp

            ; Save used registers
            push    ax

            ; get current image position (x)
            mov     ax, WORD PTR [bp+PARM]
            mov     Xpos, ax

            ; restore saved registers
            pop     ax

            ; remove frame pointer
            mov     sp, bp
            pop     bp

            ret

setxpos     endp

; ---------------------------------------------------------------------------
; SETIMAGE
;
; Set source image for ISR.
;
; Inputs:  WORD source image,
;          WORD source x,
;          WORD source y
;
; Outputs: none
; ---------------------------------------------------------------------------
            public  setimage

IFDEF mem_S
setimage    proc    near
ELSE
setimage    proc    far
ENDIF
            ; create frame pointer
            push    bp
            mov     bp, sp

            ; Save used registers
            push    ax

            ; get source image
            mov     ax, WORD PTR [bp+PARM]
            mov     Image, ax

            ; get image coordinates
            mov     ax, WORD PTR [bp+PARM+2]
            mov     SrcX, ax
            mov     ax, WORD PTR [bp+PARM+4]
            mov     SrcY, ax

            ; restore saved registers
            pop     ax

            ; remove frame pointer
            mov     sp, bp
            pop     bp

            ret

setimage    endp

; ---------------------------------------------------------------------------
; GETIRQINDEX
;
; Return the IRQ index given an IRQ number
;
; Inputs : WORD IRQ num
;            AX: valid values  : 2  , 3  , 5  , 0Ah
;
; Outputs: WORD IRQ index
;            AX: valid returns : 0Ah, 0Bh, 0Dh, 72h
; ---------------------------------------------------------------------------
            public  getirqindex

IFDEF mem_S
getirqindex proc    near
ELSE
getirqindex proc    far
ENDIF
            ; determine IRQ index based on IRQ number
            xor     ah, ah
            cmp     al, 7h
            ja      irq_cascade_index

            add     al, 8h
            jmp     done

irq_cascade_index:
            add     al, 70h
            sub     al, 8h

done:
            ret

getirqindex endp

; ---------------------------------------------------------------------------
; INITVLINEISR
;
; Chain in VLINE interrupt service routine.
;
; Inputs : WORD IRQnum
;            AX: valid values: 2, 3, 5, 0Ah
;
; Outputs: none
; ---------------------------------------------------------------------------
            public  initvlineisr

IFDEF mem_S
initvlineisr proc   near
ELSE
initvlineisr proc   far
ENDIF
            ; create frame pointer
            push    bp
            mov     bp, sp

            ; save used registers
            push    bx
            push    dx
            push    ds
            push    es

            ; get IRQ number
            mov     ax, WORD PTR [bp+PARM]
            mov     IRQnum, al
            Mcall   getirqindex
            mov     IRQindex, al

            ; save IRQ mask for master 8259 (IRQs 0 - 7)
            mov     dx, 21h
            in      al, dx
            mov     IRQmask1, al

            ; save IRQ mask for cascaded 8259 (IRQs 8 - 15)
            mov     dx, 0A1h
            in      al, dx
            mov     IRQmask2, al

            ; load IRQ number and enable interrupts in appropriate 8259
            xor     ch, ch
            mov     cl, IRQnum

            ; if 0 <= IRQ <= 7  - program master 8259
            ; if 8 <= IRQ <= 15 - program cascaded 8259
            ; bit = 0 - enable interrupt
            ; bit = 1 - disable interrupt
            mov     al, 1
            cmp     cl, 8h
            jb      shiftleft
            sub     cl, 8h

shiftleft:
            shl     al, 1
            loop    shiftleft

            cmp    IRQnum, 7h
            ja     irq_cascade

            ; enable interrupt on master 8259 for IRQ 2, 3, 5
            mov     dx, 21h
            mov     ah, 0FFh
            sub     ah, al
            mov     al, IRQmask1
            and     al, ah
            cli                     ; disable interrupts during mask update
            out     dx, al
            sti
            jmp     continue

irq_cascade:
            ; enable interrupt on cascaded 8259 for IRQ 10 - IRQ 2 also needs
            ; to be enabled

            push    ax
            ; enable cascaded interrupt 2 on master 8259
            mov     ah, 0FBh
            mov     dx, 021h        ; IRQ 2 cascades to IRQ 10
            mov     al, IRQmask1
            and     al, ah
            cli                     ; disable interrupts during mask update
            out     dx, al
            sti

            pop     ax
            ; enable interrupt 10 on cascaded 8259
            mov     dx, 0A1h
            mov     ah, 0FFh
            sub     ah, al
            mov     al, IRQmask2
            and     al, ah
            cli                     ; disable interrupts during mask update
            out     dx, al
            sti

continue:
            ; initialize variables for ISR
            mov     ax, 0
            mov     Count, al

            ; get old IRQ vector address (in ES:BX)
            mov     al, IRQindex
            mov     ah, 35h
            int     21h

            ; save vector address
            mov     ax, es
            mov     OldIntVector, bx
            mov     OldIntVector+2, ax

            ; get address of interrupt service routine (in DS:DX)
            mov     dx, offset cs:VlineISR
            mov     ax, cs
            mov     ds, ax

            ; set new vector address
            mov     al, IRQindex
            mov     ah, 25h
            int     21h

            ; restore registers
            pop     es
            pop     ds
            pop     dx
            pop     bx

            ; remove frame pointer
            mov     sp, bp
            pop     bp

            ret

initvlineisr endp

; ---------------------------------------------------------------------------
; CANCELVLINEISR
;
; Disable timer interrupt routine.
;
; Inputs : none
;
; Outputs: none
; ---------------------------------------------------------------------------
            public  cancelvlineisr

IFDEF mem_S
cancelvlineisr proc near
ELSE
cancelvlineisr proc far
ENDIF
            ; save used registers
            push    dx
            push    ds

            ; restore old IRQ vector address (in DS:DX)
            mov     dx, OldIntVector
            mov     ax, OldIntVector+2
            mov     ds, ax

            mov     al, IRQindex
            mov     ah, 25h
            int     21h

            ; restore IRQ masks - disable interrupts while updating masks
            cli
            mov     dx, 21h
            mov     al, IRQmask1
            out     dx, al
            mov     dx, 0A1h
            mov     al, IRQmask2
            out     dx, al
            sti

            ; restore registers
            pop     ds
            pop     dx

            ret

cancelvlineisr endp

; ---------------------------------------------------------------------------
; INTERRUPTS_OFF
;
; Disable system interrupts.
;
; Inputs : none
;
; Outputs: none
; ---------------------------------------------------------------------------
            public  interrupts_off

IFDEF mem_S
interrupts_off proc near
ELSE
interrupts_off proc far
ENDIF
            ; disable system interrupts
            cli

            ret

interrupts_off endp

; ---------------------------------------------------------------------------
; INTERRUPTS_ON
;
; Enable system interrupts.
;
; Inputs : none
;
; Outputs: none
; ---------------------------------------------------------------------------
            public  interrupts_on

IFDEF mem_S
interrupts_on proc near
ELSE
interrupts_on proc far
ENDIF
            ; enable system interrupts
            sti

            ret

interrupts_on endp

            end

