/*==========================================================================
  SBUFP.C

  This example code shows how to implement single buffered drawing by
  polling. Several image "frames" are loaded into off-screen memory so that
  they can blited to the screen in sequence. To prevent image tearing, a blit
  is not performed until the display reaches a specific vertical line
  (usually after the lower part of the blit). In this example, this is done
  by polling VLINE register. The frame rate is determined by the vertical
  frequency of the display.

  Copyright (c) 1994-1995 ATI Technologies Inc. All rights reserved
 =========================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <dos.h>
#include "..\util\atim64.h"
#include "..\util\sample.h"
#include "..\util\vint.h"
#include "..\util\vtga.h"

#define INSTANCE            0
#define BACKGROUND_COLOR    0x11

/* Main C program */

int main (void)
{
    char filename[20];
    TARGA_HEADER header;
    int width, height;
    int srcx, srcy;
    int savex, savey;
    int i, j;
    int y_draw;
    int frame;
    int frames;
    int framesperwidth;
    int topline;
    int step;
    POINT points[8];

    // check if Mach64 adapter is installed
    if (detect_mach64 (INSTANCE) != YES_MACH64)
    {
        printf ("mach64 based adapter was not found.\n");
        return (1);
    }

    // fill global query structure by calling Mach 64 ROM
    if (query_hardware () != NO_ERROR)
    {
        printf ("Failed ROM call to query mach64 hardware.\n");
        return (1);
    }

    // check if Mach 64 VGA controller is enabled
    if (querydata.vga_type != VGA_ENABLE)
    {
        printf ("This sample code example requires an enabled mach64 VGA "
                "controller.\n");
        return (1);
    }

    // set an accelerator mode
    if (open_mode (MODE_640x480, PITCH_XRES, COLOR_DEPTH_8) != NO_ERROR)
    {
        printf ("Error in setting display mode.\n");
        return (1);
    }

    // check if color depth is 8 bpp
    if (modeinfo.bpp != 8)
    {
        close_mode ();
        printf ("This example requires an 8 bpp mode.\n");
        return (1);
    }

    // setup engine context and clear screen
    init_engine ();
    clear_screen (0, 0, modeinfo.xres, modeinfo.yres);

    // get targa header information
    if (get_targa_header ("..\\image\\frame1.tga", &header) != SUCCESS)
    {
        close_mode ();
        printf ("Error reading targa file header information.\n");
        return (1);
    }

    // setup image size, source area, save area, and position increment
    width = header.width;
    height = header.height;
    srcx = 0;
    srcy = modeinfo.yres + height;
    savex = 0;
    savey = modeinfo.yres;
    y_draw = modeinfo.yres - height;
    step = 2;

    // determine how large to expand the scissors
    frames = 8;
    framesperwidth = modeinfo.xres / width;
    topline = frames / framesperwidth;
    if ((topline * framesperwidth) != frames)
    {
        topline++;
    }
    topline = ((topline + 1) * height) + modeinfo.yres;

    // expand scissors to include source and save areas
    wait_for_fifo (4);
    regw (SC_LEFT, 0);
    regw (SC_TOP, 0);
    regw (SC_RIGHT, modeinfo.xres - 1);
    regw (SC_BOTTOM, topline - 1);

    // draw background the same as the image background
    set_fg_color (BACKGROUND_COLOR);
    draw_rectangle (0, 0, modeinfo.xres, modeinfo.yres);

    // load source images into off-screen memory for frame blitting and
    // record their position
    frame = 0;
    i = 0;
    j = 0;
    while (frame < frames)
    {
        // record each frame coordinate
        points[frame].x = srcx + (width * i);
        points[frame].y = srcy + (height * j);

        // load next frame image into video memory
        sprintf (filename, "..\\image\\frame%d.tga", frame + 1);
        if (load_targa (filename, points[frame].x, points[frame].y) != SUCCESS)
        {
            close_mode ();
            printf ("Error loading targa file to memory.\n");
            return (1);
        }

        // adjust location of frame load coordinate as necessary
        frame++;
        i++;
        if (i > ((modeinfo.xres / header.width) - 1))
        {
            i = 0;
            j++;
        }
    }

    // set palette from targa color table (8 bpp)
    if (header.pixel_depth == 8)
    {
        if (set_targa_palette ("..\\image\\frame1.tga") != SUCCESS)
        {
            close_mode ();
            printf ("Error reading targa file color table information.\n");
            return (1);
        }
    }

    // wait for a key to start
    getch ();

    // setup engine for blits
    wait_for_fifo (3);
    regw (DP_SRC, FRGD_SRC_BLIT);
    regw (SRC_CNTL, 0);
    regw (DST_CNTL, DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT);

    // main draw loop
    i = 0;
    frame = 0;
    while (i < (modeinfo.xres - width))
    {
        // display old frame, update new frame
        poll_for_vblank ();  // poll until start of next vertical blank

        // restore previous frame (if first frame has occurred already)
        if (i > 0)
        {
            blit (savex, savey, i-step, y_draw, width, height);
        }

        // save current frame
        blit (i, y_draw, savex, savey, width, height);

        // update current frame
        blit (points[frame].x, points[frame].y, i, y_draw, width, height);

        // cycle through frames
        i = i + step;
        frame++;
        if (frame >= frames)
        {
            frame = 0;
        }
    }

    // wait for a carriage return
    getch ();

    // disable accelerator mode and switch back to VGA text mode
    close_mode ();

    return (0);
}

