; ===========================================================================
; ROMCALLS.ASM
;
; Interface routines to access MACH64 ROM services.
;
; Compiling:
;   masm /Ml /D<memory model> romcalls.asm;
;       <memory model> = mem_S for SMALL model,
;                        mem_M for MEDIUM model,
;                        mem_L for LARGE model
;
; Copyright (c) 1993-1995 ATI Technologies Inc. All rights reserved
; ===========================================================================

include atim64.inc

IFDEF mem_S
PARM        equ     4   ; passed parameters start at bp+4 for small model
ELSE
PARM        equ     6   ; passed parameters start at bp+6 for other models
ENDIF

IFDEF mem_S
.MODEL  SMALL, C
ELSEIFDEF mem_M
.MODEL  MEDIUM, C
ELSE
.MODEL  LARGE, C
ENDIF

.DATA

ati_sig     db      '761295520', 0
rom_addr    dw      64h
            dw      0C000h
            db      0

.CODE
.286

; Macro for 'call' model handling
Mcall       macro   routine
IFDEF mem_S
            call    NEAR PTR routine
ELSE
            call    FAR PTR routine
ENDIF
            endm

; Externals
IFDEF mem_S
extrn       get_rom_base:NEAR
ELSE
extrn       get_rom_base:FAR
ENDIF

; ---------------------------------------------------------------------------
; LOAD_MODE_PARMS
;
; Load accelerator mode parameters to determine the mode to be invoked by a
; following call to SET_DISPLAY_MODE. Input parameters are fetched from the
; stack. See SAMPLE.H for the symbolic values.
;
; Inputs : WORD resolution code
;            MODE_640x480   - 640x480
;            MODE_800x600   - 800x600
;            MODE_1024x768  - 1024x768
;            MODE_1280x1024 - 1280x1024
;
;          WORD pitch code
;            PITCH_1024     - 1024
;            PITCH_NOCHANGE - don't change
;            PITCH_XRES     - pitch size = resolution width
;
;          WORD deep color code
;            COLOR_DEPTH_4  - 4 bpp
;            COLOR_DEPTH_8  - 8 bpp
;            COLOR_DEPTH_15 - 15 bpp (555)
;            COLOR_DEPTH_16 - 16 bpp (565)
;            COLOR_DEPTH_24 - 24 bpp (RGB)
;            COLOR_DEPTH_32 - 32 bpp (RGBa)
;
; Outputs: Return codes
;            NO_ERROR      - no error
;            YES_ERROR     - function complete with error
;            NOT_SUPPORTED - function not supported
; ---------------------------------------------------------------------------
            public  load_mode_parms

IFDEF mem_S
load_mode_parms proc near
ELSE
load_mode_parms proc far
ENDIF

            ; create frame pointer
            push    bp
            mov     bp, sp

            ; save registers used
            push    cx

            ; setup parameters for call to ATI rom
            Mcall   get_rom_base                ; get rom segment in ax
            mov     rom_addr+2, ax
            mov     ax, WORD PTR [bp+PARM]      ; get resolution code
            mov     ch, al
            mov     ax, WORD PTR [bp+PARM+2]    ; get pitch code
            shl     al, 6
            mov     cl, al
            mov     ax, WORD PTR [bp+PARM+4]    ; get deep color code
            and     al, 7
            or      cl, al
            mov     ax, 0                       ; function code 0
            mov     rom_addr, 64h
            call    DWORD PTR rom_addr          ; call ROM

            ; setup error code in AL
            mov     al, ah
            xor     ah, ah

            ; restore saved registers
            pop     cx

            ; remove frame pointer
            mov     sp, bp
            pop     bp

            ret

load_mode_parms endp

; ---------------------------------------------------------------------------
; SET_DISPLAY_MODE
;
; Set display to accelerator (after a call to LOAD_MODE_PARMS()) or VGA mode.
; Input parameters are fetched from the stack. See SAMPLE.H for symbolic
; values.
;
; Inputs : WORD display mode
;            VGA_MODE         - VGA
;            ACCELERATOR_MODE - Accelerator
;
; Outputs: Return codes
;            NO_ERROR      - no error
;            YES_ERROR     - function complete with error
;            NOT_SUPPORTED - function not supported
; ---------------------------------------------------------------------------
            public  set_display_mode

IFDEF mem_S
set_display_mode proc near
ELSE
set_display_mode proc far
ENDIF

            ; create frame pointer
            push    bp
            mov     bp, sp

            ; save registers used
            push    cx

            ; setup parameters for call to ATI rom
            Mcall   get_rom_base                ; get rom segment in ax
            mov     rom_addr+2, ax
            mov     ax, WORD PTR [bp+PARM]      ; get display mode flag
            mov     cl, al
            mov     ax, 1                       ; function code 1
            mov     rom_addr, 64h
            call    DWORD PTR rom_addr          ; call ROM

            ; setup error code in AL
            mov     al, ah
            xor     ah, ah

            ; restore saved registers
            pop     cx

            ; remove frame pointer
            mov     sp, bp
            pop     bp

            ret

set_display_mode endp

; ---------------------------------------------------------------------------
; LOAD_AND_SET_MODE
;
; Load accelerator mode parameters and set accelerator mode. Input parameters
; are fetched from the stack. See SAMPLE.H for symbolic values.
;
; Inputs : WORD resolution code
;            MODE_640x480   - 640x480
;            MODE_800x600   - 800x600
;            MODE_1024x768  - 1024x768
;            MODE_1280x1024 - 1280x1024
;
;          WORD pitch code
;            PITCH_1024     - 1024
;            PITCH_NOCHANGE - don't change
;            PITCH_XRES     - pitch size = resolution width
;
;          WORD deep color code
;            COLOR_DEPTH_4  - 4 bpp
;            COLOR_DEPTH_8  - 8 bpp
;            COLOR_DEPTH_15 - 15 bpp (555)
;            COLOR_DEPTH_16 - 16 bpp (565)
;            COLOR_DEPTH_24 - 24 bpp (RGB)
;            COLOR_DEPTH_32 - 32 bpp (RGBa)
;
; Outputs: Return codes
;            NO_ERROR      - no error
;            YES_ERROR     - function complete with error
;            NOT_SUPPORTED - function not supported
; ---------------------------------------------------------------------------
            public  load_and_set_mode

IFDEF mem_S
load_and_set_mode proc near
ELSE
load_and_set_mode proc far
ENDIF

            ; create frame pointer
            push    bp
            mov     bp, sp

            ; save registers used
            push    cx

            ; setup parameters for call to ATI rom
            Mcall   get_rom_base                ; get rom segment in ax
            mov     rom_addr+2, ax
            mov     ax, WORD PTR [bp+PARM]      ; get resolution code
            mov     ch, al
            mov     ax, WORD PTR [bp+PARM+2]    ; get pitch code
            shl     al, 6
            mov     cl, al
            mov     ax, WORD PTR [bp+PARM+4]    ; get deep color code
            and     al, 7
            or      cl, al
            mov     ax, 2                       ; function code 2
            mov     rom_addr, 64h
            call    DWORD PTR rom_addr          ; call ROM

            ; setup error code in AL
            mov     al, ah
            xor     ah, ah

            ; restore saved registers
            pop     cx

            ; remove frame pointer
            mov     sp, bp
            pop     bp

            ret

load_and_set_mode endp

; ---------------------------------------------------------------------------
; GET_QUERY_SIZE
;
; Retrieve the query size in bytes to reserve for a subsequent call to
; FILL_QUERY_STRUCTURE. The size of the header or header & mode tables can be
; retrieved. Input parameters are fetched from the stack. See SAMPLE.H for
; symbolic values.
;
; Inputs : WORD information type
;            HEADER_ONLY     - header information only
;            HEADER_AND_MODE - header and mode table information
;
; Outputs: Returns size in bytes
;            size in bytes
;               or
;            0 if the function returns with an error
; ---------------------------------------------------------------------------
            public  set_display_mode

IFDEF mem_S
get_query_size proc near
ELSE
get_query_size proc far
ENDIF

            ; create frame pointer
            push    bp
            mov     bp, sp

            ; save registers used
            push    cx

            ; setup parameters for call to ATI rom
            Mcall   get_rom_base                ; get rom segment in ax
            mov     rom_addr+2, ax
            mov     ax, WORD PTR [bp+PARM]      ; get info type flag
            mov     cl, al
            mov     ax, 8                       ; function code 8
            mov     rom_addr, 64h
            call    DWORD PTR rom_addr          ; call ROM

            ; check for errors in AH
            cmp     ah, 0
            je      no_error

            ; error: set ax to zero
            mov     ax, 0
            jmp     qsize_exit

no_error:
            ; no error: move size from cx to ax
            mov     ax, cx

qsize_exit:
            ; restore saved registers
            pop     cx

            ; remove frame pointer
            mov     sp, bp
            pop     bp

            ret

get_query_size endp

; ---------------------------------------------------------------------------
; FILL_QUERY_STRUCTURE
;
; Fill a query structure containing information about the installed hardware.
; This function should be called after called GET_QUERY_SIZE and with the
; same information type flag. This insures that the structure is large
; enough to hold the fill information. Input parameters are fetched from the
; stack. See SAMPLE.H for symbolic values.
;
; Inputs : WORD information type
;            HEADER_ONLY     - header information only
;            HEADER_AND_MODE - header and mode table information
;
;          FAR POINTER address of structure to be filled (DX:BX)
;
; Outputs: Returns error code in ax
;            NO_ERROR      - no error
;            YES_ERROR     - function complete with error
;            NOT_SUPPORTED - function not supported
; ---------------------------------------------------------------------------
            public  set_display_mode

IFDEF mem_S
fill_query_structure proc near
ELSE
fill_query_structure proc far
ENDIF

            ; create frame pointer
            push    bp
            mov     bp, sp

            ; save registers used
            push    bx
            push    cx
            push    dx

            ; setup parameters for call to ATI rom
            Mcall   get_rom_base                ; get rom segment in ax
            mov     rom_addr+2, ax
            mov     cx, WORD PTR [bp+PARM]      ; get info type flag
            mov     bx, WORD PTR [bp+PARM+2]    ; get offset of address
            mov     dx, WORD PTR [bp+PARM+4]    ; get segment of address
            mov     ax, 9                       ; function code 9
            mov     rom_addr, 64h
            call    DWORD PTR rom_addr          ; call ROM

            ; setup error code in AL
            mov     al, ah
            xor     ah, ah

            ; restore saved registers
            pop     dx
            pop     cx
            pop     bx

            ; remove frame pointer
            mov     sp, bp
            pop     bp

            ret

fill_query_structure endp


; ---------------------------------------------------------------------------
; GET_IO_BASE
; ---------------------------------------------------------------------------
            public  get_io_base

IFDEF mem_S
get_io_base proc near
ELSE
get_io_base proc far
ENDIF

            ; create frame pointer
            push    bp
            mov     bp, sp

            ; save registers used
            push    bx
            push    cx
            push    dx
            push    es
            push    di

            ; setup parameters for call to ATI rom
            Mcall   get_rom_base                ; get rom segment in ax
            mov     rom_addr+2, ax
            mov     ax, 12h                     ; function code 12
            mov     rom_addr, 64h
            call    DWORD PTR rom_addr          ; call ROM

            les     di, DWORD PTR [bp+PARM]
            mov     WORD PTR es:[di], dx
            mov     WORD PTR es:[di+2], cx
            mov     ax, 1
            jmp     exit_get_io_base

get_io_base_error:
            mov     ax, 0

exit_get_io_base:

            ; restore saved registers
            pop     di
            pop     es
            pop     dx
            pop     cx
            pop     bx

            ; remove frame pointer
            mov     sp, bp
            pop     bp

            ret

get_io_base endp

            end
