/*===========================================================================
 PGL.H - Copyright (c) 1994 ATI Technologies Inc.  All rights reserved

 PGL function prototype include file. Include in source that uses the PGL.
 ===========================================================================*/

// define PGL_H for other files that include PGL.H
#ifndef PGL_H
#define PGL_H

// --------------------------------------------------------------------------
// PGL function input constants
// --------------------------------------------------------------------------

// 'fill_type' constants for PGL_getquerysize() and PGL_queryhardware()
#define HEADER_ONLY         0           // query
#define HEADER_AND_MODE     1

// 'register_access' constants for PGL_initaperture()
#define USE_VGA_APERTURE    0           // aperture
#define USE_LINEAR_APERTURE 1
#define AUTO_APERTURE       2

// 'attribute_flags' constants for PGL_initmode()
#define NO_ATTRIBUTES       0           // mode setting
#define PALETTIZED_MODE     1
#define BYPASS_MODE         2
#define CLOSE_MODE          4
#define NO_CLEAR_SCREEN     8
#define NO_PACKED_PIXEL     1024
#define NO_AUTO_DEPTH       2048

#define NO_PITCH_SET        NO_ATTRIBUTES   // backwards compatibility

// 'tableinfo' constant for PGL_initmode()
#define NO_CRT_TABLE        NULL

// 'eeprom_offset' constant for PGL_initmode()
#define NO_EEPROM_OFFSET    -1

// 'mode_number' constants for PGL_initmode(); also standard ROM mode numbers
#define MODE_640x480        0x12
#define MODE_800x600        0x6a
#define MODE_1024x768       0x55
#define MODE_1280x1024      0x83
#define MODE_1600x1200      0x84
#define MODE_OTHER          0x82        // usually 1152x864
#define USE_CRT_TABLE       0x81        // use if 'tableinfo' is valid
#define USE_EEPROM_OFFSET   0x80        // use if 'eeprom_offset is valid

// 'depth' constants for PGL_initmode() - ignored if NO_AUTO_DEPTH flag set
#define DEPTH_AUTO          0           // default choice
#define DEPTH_24_RGB        0
#define DEPTH_24_BGR        1
#define DEPTH_32_RGBA       0
#define DEPTH_32_ARGB       1
#define DEPTH_32_BGRA       2
#define DEPTH_32_ABGR       3

// Vga font size constants for PGL_vgafont()
#define VGAFONT_8x8         0
#define VGAFONT_8x14        1
#define VGAFONT_8x16        2

// Input constants for PGL_setfifocontrol()
#define FIFO_CHECK_NORMAL     0
#define FIFO_CHECK_TIMEOUT    1
#define FIFO_ACTION_NONE      0
#define FIFO_ACTION_TERMINATE 1

// 'mix' constants for PGL_setfgmix() and PGL_setbgmix()
#define NOT_D_MIX           0
#define ZERO_MIX            1
#define ONE_MIX             2
#define D_MIX               3
#define NOT_S_MIX           4
#define D_XOR_S_MIX         5
#define NOT_D_XOR_S_MIX     6
#define S_MIX               7
#define NOT_D_OR_NOT_S_MIX  8
#define D_OR_NOT_S_MIX      9
#define NOT_D_OR_S_MIX      10
#define D_OR_S_MIX          11
#define D_AND_S_MIX         12
#define NOT_D_AND_S_MIX     13
#define D_AND_NOT_S_MIX     14
#define NOT_D_AND_NOT_S_MIX 15
#define D_PLUS_S_DIV2_MIX   23

// Line 'type' constants PGL_setlinetype()
#define USERDEFINED_LINE    0
#define DOTTED_LINE         1
#define SHORT_DASHED_LINE   2
#define DASH_DOT_LINE       3
#define DOUBLE_DOTTED_LINE  4
#define LONG_DASH_LINE      5
#define DASH_DOUBLE_DOT_LINE 6
#define SOLID_LINE          7
#define INVISIBLE_LINE      8

// Fill 'type' constants for PGL_setfilltype()
#define USERDEFINED_FILL    0
#define SOLID_FILL          1

// 'saveflag' constants for PGL_ncscreentoscreenblit()
#define SAVE_BLIT           0
#define RESTORE_BLIT        1

// 'generic_color' constants for PGL_getcolorcode()
#define BLACK               0
#define DARKBLUE            1
#define DARKGREEN           2
#define DARKCYAN            3
#define DARKRED             4
#define DARKMAGENTA         5
#define BROWN               6
#define LIGHTGRAY           7
#define DARKGRAY            8
#define LIGHTBLUE           9
#define LIGHTGREEN          10
#define LIGHTCYAN           11
#define LIGHTRED            12
#define LIGHTMAGENTA        13
#define YELLOW              14
#define WHITE               15

// 'direction' constant for pgl_movemem(), PGL_writeblock(), PGL_readblock()
#define MEM_READ    0
#define MEM_WRITE   1

// 'table_selection' constants for PGL_getcrttable()
#define CRT_640x480_60      0
#define CRT_640x480_72      1
#define CRT_640x480_75      2
#define CRT_800x600_56      3
#define CRT_800x600_60      4
#define CRT_800x600_72      5
#define CRT_800x600_75      6
#define CRT_1024x768_60     7
#define CRT_1024x768_70     8
#define CRT_1024x768_75     9
#define CRT_1280x1024_75    10

// --------------------------------------------------------------------------
// PGL function output constants
// --------------------------------------------------------------------------

// General return codes for most functions (some functions may use NO_ERROR
// but use other constants to describe error conditions such as
// PGL_initaperture())
#define NO_ERROR            0
#define YES_ERROR           1

// Return codes shared by PGL_initaperture() and PGL_initmode()
#define MODE_SUPPORTED      0           // PGL_initmode() only
#define MODE_NOT_SUPPORTED  1           // PGL_initmode() only
#define NO_APERTURE         2
#define QUERY_FAILED        3           // PGL_initaperture() only
#define FAILED_REG_TEST     4           // PGL_initaperture() only

// Error return code for PGL_makecolor() for no color match in 4 or 8 bpp
#define NOCOLORMATCH        0xffff

// --------------------------------------------------------------------------
// Miscellaneous constants
// --------------------------------------------------------------------------

// Fifo and idle constants for PGL_waitforidle(), PGL_waitforemptyfifo(),
// and PGL_waitforfifo() when the timeout feature is active -- see function
// PGL_setfifocontrol()
#define DOS_TICK_ADDRESS    0x0000046c
#define IDLE_TIMEOUT        50
#define FIFO_TIMEOUT        50
#define WAITFOREMPTYFIFO    0
#define WAITFORFIFO         1
#define WAITFORIDLE         2

// DAC BYTE index constants for ioDAC_REGS register -- can be used with
// ior8() or iow8()
#define DAC_W_INDEX         0
#define DAC_DATA            1
#define DAC_MASK            2
#define DAC_R_INDEX         3

// Maximum engine dimensions used in the PGL
#define ENGINE_MIN_X        0
#define ENGINE_MIN_Y        0
#define ENGINE_MAX_X        4095
#define ENGINE_MAX_Y        16383

// Status codes used in PGL_modeconfig structure
#define LINEAR_APERTURE_DISABLED  0
#define LINEAR_APERTURE_ENABLED   1
#define VGA_APERTURE_DISABLED     0
#define VGA_APERTURE_ENABLED      1

// Color codes for 4 & 8 bpp modes - palette entry based
#define DARKBLUE8           0x00000001
#define DARKGREEN8          0x00000002
#define DARKCYAN8           0x00000003
#define DARKRED8            0x00000004
#define DARKMAGENTA8        0x00000005
#define BROWN8              0x00000006
#define LIGHTGRAY8          0x00000007
#define DARKGRAY8           0x00000008
#define LIGHTBLUE8          0x00000009
#define LIGHTGREEN8         0x0000000a
#define LIGHTCYAN8          0x0000000b
#define LIGHTRED8           0x0000000c
#define LIGHTMAGENTA8       0x0000000d
#define YELLOW8             0x0000000e
#define WHITE8              0x0000000f
#define HIWHITE8            0x000000ff

// Color codes for 15 bpp (16 bpp 555 color weight) modes - direct mapped
#define DARKBLUE15          0x00000014
#define DARKGREEN15         0x00000280
#define DARKCYAN15          0x00000294
#define DARKRED15           0x00005000
#define DARKMAGENTA15       0x00005014
#define BROWN15             0x00005280
#define LIGHTGRAY15         0x00005294
#define DARKGRAY15          0x0000294a
#define LIGHTBLUE15         0x0000001f
#define LIGHTGREEN15        0x000003e0
#define LIGHTCYAN15         0x000003ff
#define LIGHTRED15          0x00007c00
#define LIGHTMAGENTA15      0x00007c1f
#define YELLOW15            0x00007fe0
#define WHITE15             0x0000ffff

// Color codes for 16 bpp (16 bpp 565 color weight) modes - direct mapped
#define DARKBLUE16          0x00000014
#define DARKGREEN16         0x000004c0
#define DARKCYAN16          0x000004d4
#define DARKRED16           0x0000a000
#define DARKMAGENTA16       0x0000a014
#define BROWN16             0x0000a4c0
#define LIGHTGRAY16         0x0000a4d4
#define DARKGRAY16          0x0000528a
#define LIGHTBLUE16         0x0000001f
#define LIGHTGREEN16        0x000007e0
#define LIGHTCYAN16         0x000007ff
#define LIGHTRED16          0x0000f800
#define LIGHTMAGENTA16      0x0000f81f
#define YELLOW16            0x0000ffe0
#define WHITE16             0x0000ffff

// Color codes for 24 bpp RGB color weight modes - direct mapped
#define DARKBLUE24_RGB      0x0000009e
#define DARKGREEN24_RGB     0x00009e00
#define DARKCYAN24_RGB      0x00009e9e
#define DARKRED24_RGB       0x009e0000
#define DARKMAGENTA24_RGB   0x009e009e
#define BROWN24_RGB         0x009e9e00
#define LIGHTGRAY24_RGB     0x009e9e9e
#define DARKGRAY24_RGB      0x00555555
#define LIGHTBLUE24_RGB     0x000000ff
#define LIGHTGREEN24_RGB    0x0000ff00
#define LIGHTCYAN24_RGB     0x0000ffff
#define LIGHTRED24_RGB      0x00ff0000
#define LIGHTMAGENTA24_RGB  0x00ff00ff
#define YELLOW24_RGB        0x00ffff00
#define WHITE24_RGB         0x00ffffff

// Color codes for 24 bpp BGR color weight modes - direct mapped
#define DARKBLUE24_BGR      0x009e0000
#define DARKGREEN24_BGR     0x00009e00
#define DARKCYAN24_BGR      0x009e9e00
#define DARKRED24_BGR       0x0000009e
#define DARKMAGENTA24_BGR   0x009e009e
#define BROWN24_BGR         0x00009e9e
#define LIGHTGRAY24_BGR     0x009e9e9e
#define DARKGRAY24_BGR      0x00555555
#define LIGHTBLUE24_BGR     0x00ff0000
#define LIGHTGREEN24_BGR    0x0000ff00
#define LIGHTCYAN24_BGR     0x00ffff00
#define LIGHTRED24_BGR      0x000000ff
#define LIGHTMAGENTA24_BGR  0x00ff00ff
#define YELLOW24_BGR        0x0000ffff
#define WHITE24_BGR         0x00ffffff

// Color codes for 32 bpp RGBa color weight modes - direct mapped
#define DARKBLUE32_RGBA     0x00009e00
#define DARKGREEN32_RGBA    0x009e0000
#define DARKCYAN32_RGBA     0x009e9e00
#define DARKRED32_RGBA      0x9e000000
#define DARKMAGENTA32_RGBA  0x9e009e00
#define BROWN32_RGBA        0x9e9e0000
#define LIGHTGRAY32_RGBA    0x9e9e9e00
#define DARKGRAY32_RGBA     0x55555500
#define LIGHTBLUE32_RGBA    0x0000ff00
#define LIGHTGREEN32_RGBA   0x00ff0000
#define LIGHTCYAN32_RGBA    0x00ffff00
#define LIGHTRED32_RGBA     0xff000000
#define LIGHTMAGENTA32_RGBA 0xff00ff00
#define YELLOW32_RGBA       0xffff0000
#define WHITE32_RGBA        0xffffff00

// Color codes for 32 bpp aBGR color weight modes - direct mapped
#define DARKBLUE32_ABGR     0x009e0000
#define DARKGREEN32_ABGR    0x00009e00
#define DARKCYAN32_ABGR     0x009e9e00
#define DARKRED32_ABGR      0x0000009e
#define DARKMAGENTA32_ABGR  0x009e009e
#define BROWN32_ABGR        0x00009e9e
#define LIGHTGRAY32_ABGR    0x009e9e9e
#define DARKGRAY32_ABGR     0x00555555
#define LIGHTBLUE32_ABGR    0x00ff0000
#define LIGHTGREEN32_ABGR   0x0000ff00
#define LIGHTCYAN32_ABGR    0x00ffff00
#define LIGHTRED32_ABGR     0x000000ff
#define LIGHTMAGENTA32_ABGR 0x00ff00ff
#define YELLOW32_ABGR       0x0000ffff
#define WHITE32_ABGR        0x00ffffff

// Color codes for 32 bpp aRGB color weight modes - direct mapped
#define DARKBLUE32_ARGB     0x0000009e
#define DARKGREEN32_ARGB    0x00009e00
#define DARKCYAN32_ARGB     0x00009e9e
#define DARKRED32_ARGB      0x009e0000
#define DARKMAGENTA32_ARGB  0x009e009e
#define BROWN32_ARGB        0x009e9e00
#define LIGHTGRAY32_ARGB    0x009e9e9e
#define DARKGRAY32_ARGB     0x00555555
#define LIGHTBLUE32_ARGB    0x000000ff
#define LIGHTGREEN32_ARGB   0x0000ff00
#define LIGHTCYAN32_ARGB    0x0000ffff
#define LIGHTRED32_ARGB     0x00ff0000
#define LIGHTMAGENTA32_ARGB 0x00ff00ff
#define YELLOW32_ARGB       0x00ffff00
#define WHITE32_ARGB        0x00ffffff

// Color codes for 32 bpp BGRa color weight modes - direct mapped
#define DARKBLUE32_BGRA     0x9e000000
#define DARKGREEN32_BGRA    0x009e0000
#define DARKCYAN32_BGRA     0x9e9e0000
#define DARKRED32_BGRA      0x00009e00
#define DARKMAGENTA32_BGRA  0x9e009e00
#define BROWN32_BGRA        0x009e9e00
#define LIGHTGRAY32_BGRA    0x9e9e9e00
#define DARKGRAY32_BGRA     0x55555500
#define LIGHTBLUE32_BGRA    0xff000000
#define LIGHTGREEN32_BGRA   0x00ff0000
#define LIGHTCYAN32_BGRA    0xffff0000
#define LIGHTRED32_BGRA     0x0000ff00
#define LIGHTMAGENTA32_BGRA 0xff00ff00
#define YELLOW32_BGRA       0x00ffff00
#define WHITE32_BGRA        0xffffff00

// Hardware cursor bitmap dimensions
#define HWCURWIDTH          8    // width of hwcursor in words
#define HWCURHEIGHT         64   // height of hwcursor in scan lines

// --------------------------------------------------------------------------
// Mach64 engine bit constants - these are typically ORed together and are
// used with regr(), regw(), ior8(), iow8(), ior16(), iow16(), ior32(), and
// iow32(). GUI registers are only memory mapped.
// --------------------------------------------------------------------------

// CRTC_INT_CNTL register constants
#define VBLANK_OFF              0
#define VBLANK_ON               1

// GEN_TEST_CNTL register constants
#define HWCURSOR_ENABLE         0x80
#define GUI_ENGINE_ENABLE       0x100
#define BLOCKWRITE_ENABLE       0x200

// CONFIG_CNTL register constants
#define APERTURE_4M_ENABLE      1
#define APERTURE_8M_ENABLE      2
#define VGA_APERTURE_ENABLE     4

// DST_CNTL register constants
#define DST_X_RIGHT_TO_LEFT     0
#define DST_X_LEFT_TO_RIGHT     1
#define DST_Y_BOTTOM_TO_TOP     0
#define DST_Y_TOP_TO_BOTTOM     2
#define DST_X_MAJOR             0
#define DST_Y_MAJOR             4
#define DST_X_TILE              8
#define DST_Y_TILE              0x10
#define DST_LAST_PEL            0x20
#define DST_POLYGON_ENABLE      0x40
#define DST_24_ROTATION_ENABLE  0x80

// SRC_CNTL register constants
#define SRC_PATTERN_ENABLE      1
#define SRC_ROTATION_ENABLE     2
#define SRC_LINEAR_ENABLE       4
#define SRC_BYTE_ALIGN          8
#define SRC_LINE_X_RIGHT_TO_LEFT 0
#define SRC_LINE_X_LEFT_TO_RIGHT 0x10

// HOST_CNTL register constants
#define HOST_BYTE_ALIGN         1

// PAT_CNTL register constants
#define PAT_MONO_8x8_ENABLE     1
#define PAT_CLR_4x2_ENABLE      2
#define PAT_CLR_8x1_ENABLE      4

// DP_PIX_WIDTH register constants
#define DST_1BPP                0
#define DST_4BPP                1
#define DST_8BPP                2
#define DST_15BPP               3
#define DST_16BPP               4
#define DST_32BPP               6
#define SRC_1BPP                0
#define SRC_4BPP                0x100
#define SRC_8BPP                0x200
#define SRC_15BPP               0x300
#define SRC_16BPP               0x400
#define SRC_32BPP               0x600
#define HOST_1BPP               0
#define HOST_4BPP               0x10000
#define HOST_8BPP               0x20000
#define HOST_15BPP              0x30000
#define HOST_16BPP              0x40000
#define HOST_32BPP              0x60000
#define BYTE_ORDER_MSB_TO_LSB   0
#define BYTE_ORDER_LSB_TO_MSB   0x1000000

// DP_MIX register constants
#define BKGD_MIX_NOT_D              0
#define BKGD_MIX_ZERO               1
#define BKGD_MIX_ONE                2
#define BKGD_MIX_D                  3
#define BKGD_MIX_NOT_S              4
#define BKGD_MIX_D_XOR_S            5
#define BKGD_MIX_NOT_D_XOR_S        6
#define BKGD_MIX_S                  7
#define BKGD_MIX_NOT_D_OR_NOT_S     8
#define BKGD_MIX_D_OR_NOT_S         9
#define BKGD_MIX_NOT_D_OR_S         10
#define BKGD_MIX_D_OR_S             11
#define BKGD_MIX_D_AND_S            12
#define BKGD_MIX_NOT_D_AND_S        13
#define BKGD_MIX_D_AND_NOT_S        14
#define BKGD_MIX_NOT_D_AND_NOT_S    15
#define BKGD_MIX_D_PLUS_S_DIV2      0x17
#define FRGD_MIX_NOT_D              0
#define FRGD_MIX_ZERO               0x10000
#define FRGD_MIX_ONE                0x20000
#define FRGD_MIX_D                  0x30000
#define FRGD_MIX_NOT_S              0x40000
#define FRGD_MIX_D_XOR_S            0x50000
#define FRGD_MIX_NOT_D_XOR_S        0x60000
#define FRGD_MIX_S                  0x70000
#define FRGD_MIX_NOT_D_OR_NOT_S     0x80000
#define FRGD_MIX_D_OR_NOT_S         0x90000
#define FRGD_MIX_NOT_D_OR_S         0xa0000
#define FRGD_MIX_D_OR_S             0xb0000
#define FRGD_MIX_D_AND_S            0xc0000
#define FRGD_MIX_NOT_D_AND_S        0xd0000
#define FRGD_MIX_D_AND_NOT_S        0xe0000
#define FRGD_MIX_NOT_D_AND_NOT_S    0xf0000
#define FRGD_MIX_D_PLUS_S_DIV2      0x170000

// DP_SRC register constants
#define BKGD_SRC_BKGD_CLR           0
#define BKGD_SRC_FRGD_CLR           1
#define BKGD_SRC_HOST               2
#define BKGD_SRC_BLIT               3
#define BKGD_SRC_PATTERN            4
#define FRGD_SRC_BKGD_CLR           0
#define FRGD_SRC_FRGD_CLR           0x100
#define FRGD_SRC_HOST               0x200
#define FRGD_SRC_BLIT               0x300
#define FRGD_SRC_PATTERN            0x400
#define MONO_SRC_ONE                0
#define MONO_SRC_PATTERN            0x10000
#define MONO_SRC_HOST               0x20000
#define MONO_SRC_BLIT               0x30000

// CLR_CMP_CNTL register constants
#define COMPARE_FALSE               0
#define COMPARE_TRUE                1
#define COMPARE_NOT_EQUAL           4
#define COMPARE_EQUAL               5
#define COMPARE_DESTINATION         0
#define COMPARE_SOURCE              0x1000000

// FIFO_STAT register constants
#define FIFO_EMPTY                  0
#define FIFO_ERR                    0x80000000

// CONTEXT_LOAD_CNTL constants
#define CONTEXT_NO_LOAD             0
#define CONTEXT_LOAD                0x10000
#define CONTEXT_LOAD_AND_DO_FILL    0x20000
#define CONTEXT_LOAD_AND_DO_LINE    0x30000
#define CONTEXT_EXECUTE             0
#define CONTEXT_CMD_DISABLE         0x80000000

// GUI_STAT register constants
#define ENGINE_IDLE                 0
#define ENGINE_BUSY                 1
#define SCISSOR_LEFT_FLAG           0x10
#define SCISSOR_RIGHT_FLAG          0x20
#define SCISSOR_TOP_FLAG            0x40
#define SCISSOR_BOTTOM_FLAG         0x80

// --------------------------------------------------------------------------
// Status constants for PGL structures
// --------------------------------------------------------------------------

// Mach64 'chip_type' id constants (PGL_modeconfig)
#define CHIP_GX_ID          0x00d7
#define CHIP_CX_ID          0x0057
#define CHIP_CT_ID          0x4354
#define CHIP_DT_ID          0x4454

// 'chip_attribute' special CHIP handling constants (PGL_modeconfig)
#define CHIP_SP_STANDARD    0
#define CHIP_SP_GXE_FLOAT   1

// 'dac_attribute' special DAC handling constants (PGL_modeconfig)
#define DAC_SP_STANDARD     0
#define DAC_SP_ATI68860     1
#define DAC_SP_TVP3026      2
#define DAC_SP_IBM514       3

// --------------------------------------------------------------------------
// Status constants for ROM function structures
// --------------------------------------------------------------------------

// 'bus_type' constants from ROM query header (PGL_queryheader)
#define BUS_ISA                 0
#define BUS_EISA                1
#define BUS_VLB_NM              5        // non-multiplexed VLB
#define BUS_VLB                 6        // multiplexed VLB
#define BUS_PCI                 7

// 'dac_type' constants from ROM query header (PGL_queryheader)
// -lower 4 bits is DAC main type
// -upper 4 bits is DAC sub type
// -DACs other than TVP3026 will have a different sub type than 7
//
#define DAC_INTERNAL            0x00     // CT, DT internal DAC
#define DAC_IBM514              0x01
#define DAC_ATI68875            0x02     // also TLC34075
#define DAC_TVP3026_1           0x72     // same as DAC_TVP3026_2
#define DAC_BT476               0x03     // also BT478
#define DAC_BT481               0x04
#define DAC_ATT20C490           0x14     // also ATT20C491/493, SC15025/15026,
                                         //      IMS-G174, MU9C4910/1880
#define DAC_ATI68860_1          0x05     // rev B
#define DAC_ATI68860_2          0x15     // rev C
#define DAC_TVP3026_2           0x75     // same as DAC_TVP3026_1
#define DAC_STG1700             0x06
#define DAC_ATT20C498           0x16
#define DAC_STG1702             0x07
#define DAC_SC15021             0x17
#define DAC_ATT21C498           0x27
#define DAC_STG1703             0x37
#define DAC_CH8398              0x47

// 'color_depth_support' constants from ROM query header (PGL_queryheader)
#define COLORSUPPORT_16         1
#define COLORSUPPORT_15         2
#define COLORSUPPORT_24_RGB     4
#define COLORSUPPORT_24_BGR     8
#define COLORSUPPORT_32_ABGR    16
#define COLORSUPPORT_32_RGBA    32
#define COLORSUPPORT_32_BGRA    64
#define COLORSUPPORT_32_ARGB    128

// 'vga_type' constants from ROM query header (PGL_queryheader)
#define VGA_ENABLE              1
#define VGA_DISABLE             0

// 'rom_support' constants returned from ROM_DDC_query()
#define ROM_SUPPORT_DDC1        0x01
#define ROM_SUPPORT_DDC2        0x02
#define ROM_SUPPORT_DETAILED    0x40
#define ROM_SUPPORT_EDID_SETUP  0x80

// 'monitor_support' constants returned from ROM_DDC_query()
#define MONITOR_SUPPORT_NO_DDC  0x00
#define MONITOR_SUPPORT_DDC1    0x01
#define MONITOR_SUPPORT_DDC2    0x02

// 'ETiming1' constants returned in PGL_DDC_EDID structure from function
// ROM_DDC_read_EDID()
#define ETIMING1_800x600_60     0x01    // DDC EDID packet Etiming1 def
#define ETIMING1_800x600_56     0x02    // VESA modes only
#define ETIMING1_640x480_75     0x04
#define ETIMING1_640x480_72     0x08

// 'ETiming2' constants returned in PGL_DDC_EDID structure from function
// ROM_DDC_read_EDID()
#define ETIMING2_1280x1024_75   0x01    // DDC EDID packet Etiming2 def
#define ETIMING2_1024x768_75    0x02    // VESA modes only
#define ETIMING2_1024x768_70    0x04
#define ETIMING2_1024x768_60    0x08
#define ETIMING2_800x600_75     0x40
#define ETIMING2_800x600_72     0x80

// 'io_type' constants returned from ROM_get_io_base()
#define IO_TYPE_STANDARD        0
#define IO_TYPE_FLOAT           1

// --------------------------------------------------------------------------
// PGL ROM function input constants
// --------------------------------------------------------------------------

// 'pitch_code' constants for ROM_load_mode_parms() and
// ROM_load_and_set_mode()
#define PITCH_1024          0
#define PITCH_NOCHANGE      1
#define PITCH_XRES          2

// 'color_depth_code' constants for ROM_load_mode_parms() and
// ROM_load_and_set_mode()
#define COLOR_DEPTH_4       1
#define COLOR_DEPTH_8       2
#define COLOR_DEPTH_15      3
#define COLOR_DEPTH_16      4
#define COLOR_DEPTH_24      5
#define COLOR_DEPTH_32      6

// 'display_mode' constants for ROM_set_display_mode()
#define ACCELERATOR_MODE    1
#define VGA_MODE            0

// 'palettized_flag' constants for ROM_load_mode_parms(),
// ROM_set_display_mode() and PGL_load_and_set_mode()
#define PALETTIZE_ENABLE    1
#define PALETTIZE_DISABLE   0

// 'aperture_control' constants for ROM_aperture_control()
#define DISABLE_APERTURE           0    // these may be ORed together
#define ENABLE_LINEAR_APERTURE     1
#define ENABLE_VGA_APERTURE        4

// 'service_code' constants for ROM_set_dpms_control()
#define DPMS_ACTIVE         0
#define DPMS_STANDBY        1
#define DPMS_SUSPEND        2
#define DPMS_OFF            3

// --------------------------------------------------------------------------
// PGL ROM function output constants
// --------------------------------------------------------------------------

// Return codes for all ROM functions
#define NOERROR             0
#define YESERROR            1
#define NOSUPPORT           2

// 'doubleclock_flag' return constants for ROM_setdisplaymode()
#define STANDARD_CLOCK      0
#define DOUBLE_CLOCK        1

// --------------------------------------------------------------------------
// PGL defined structure types
// --------------------------------------------------------------------------

/* ensure that all structures are BYTE aligned -- same as /Zp1 CL option */
#pragma pack(1)

/* global user structures */

typedef struct                      /* PGL version structure */
{
    unsigned char minor;
    unsigned char major;
} PGL_version;

typedef struct                      /* ROM query header structure */
{
    unsigned int  size;
    unsigned char revision;
    unsigned char mode_tables;
    unsigned int  mode_table_offset;
    unsigned char mode_table_size;
    unsigned char vga_type;         // See VGA types
    unsigned int  asic_id;
    unsigned char vga_boundary;
    unsigned char memory_size;
    unsigned char dac_type;         // See DAC types
    unsigned char memory_type;
    unsigned char bus_type;         // See BUS types
    unsigned char monitor_cntl;
    unsigned int  aperture_addr;
    unsigned char aperture_cfg;
    unsigned char color_depth_support;
    unsigned char ramdac_feature_support;
    unsigned char reserved1;
    unsigned int  reserved2;
    unsigned int  io_base;
    unsigned char reserved3[6];
} PGL_queryheader;

typedef struct                      /* ROM short query structure */
{
    unsigned char aperture_cfg;
    unsigned int  aperture_addr;
    unsigned char color_depth_support;
    unsigned char memory_size;
    unsigned int  asic_id;
} PGL_shortquery;

typedef struct                      /* ROM I/O base query structure */
{
    unsigned int  io_base;
    unsigned int  io_type;
} PGL_iobasequery;

typedef struct                      /* ROM query mode table structure */
{
    unsigned int  xres;
    unsigned int  yres;
    unsigned char max_bpp;
    unsigned char mode_number;
    unsigned int  eeprom_offset;
    unsigned int  reserved1;
    unsigned int  reserved2;
    unsigned int  crt_parms1;
    unsigned char crtc_h_total;
    unsigned char crtc_h_disp;
    unsigned char crtc_h_sync_strt;
    unsigned char crtc_h_sync_wid;
    unsigned int  crtc_v_total;
    unsigned int  crtc_v_disp;
    unsigned int  crtc_v_sync_strt;
    unsigned char crtc_v_sync_wid;
    unsigned char clock_cntl;
    unsigned int  dot_clock;
    unsigned int  crt_parms2;
    unsigned char ovr_wid_top;
    unsigned char ovr_wid_bottom;
    unsigned char ovr_clr_8;
    unsigned char ovr_clr_b;
    unsigned char ovr_clr_r;
    unsigned char ovr_clr_g;
    unsigned int  reserved3;
} PGL_modetable;

typedef struct                      /* ROM mode set CRT table structure */
{
    unsigned char select2;
    unsigned char select1;
    unsigned char table_flag;       // set to 0x80 for accelerator
    unsigned char video_mode_select;
    unsigned int  crt_parms1;
    unsigned char crtc_h_total;
    unsigned char crtc_h_disp;
    unsigned char crtc_h_sync_strt;
    unsigned char crtc_h_sync_wid;
    unsigned int  crtc_v_total;
    unsigned int  crtc_v_disp;
    unsigned int  crtc_v_sync_strt;
    unsigned char crtc_v_sync_width;
    unsigned char clock_cntl;
    unsigned int  dot_clock;
    unsigned int  crt_parms2;
    unsigned char ovr_wid_top;
    unsigned char ovr_wid_bottom;
    unsigned char ovr_clr_8;
    unsigned char ovr_clr_b;
    unsigned char ovr_clr_r;
    unsigned char ovr_clr_g;
    unsigned int  reserved2;
} PGL_crttable;

typedef struct                      /* ROM DDC EDIC packet structure */
{
    unsigned char header[8];        // header - 8 bytes
    unsigned int  ID_vendor_name;   // vendor/product info - 10 bytes
    unsigned int  ID_product_code;
    unsigned long ID_serial_number;
    unsigned char make_week;
    unsigned char make_year;
    unsigned char EDID_version;     // EDID version/revision - 2 bytes
    unsigned char EDID_revision;
    unsigned char video_input_def;  // Basic display parameter/features - 5 bytes
    unsigned char max_H_image_size;
    unsigned char max_V_image_size;
    unsigned char display_gamma;
    unsigned char DPMS_support;
    unsigned char RedGreenLowBits;  // Color characteristics - 10 bytes
    unsigned char BlueWhiteLowBits;
    unsigned char Redx;
    unsigned char Redy;
    unsigned char Greenx;
    unsigned char Greeny;
    unsigned char Bluex;
    unsigned char Bluey;
    unsigned char Whitex;
    unsigned char Whitey;
    unsigned char ETiming1;         // Established timing table support
    unsigned char ETiming2;         // - 3 bytes
    unsigned char ESpecial;
    unsigned int  STiming1;         // Standard timing ID - 16 bytes
    unsigned int  STiming2;
    unsigned int  STiming3;
    unsigned int  STiming4;
    unsigned int  STiming5;
    unsigned int  STiming6;
    unsigned int  STiming7;
    unsigned int  STiming8;
    unsigned char DTiming1[18];     // Detailed timing descriptions
    unsigned char DTiming2[18];
    unsigned char DTiming3[18];
    unsigned char DTiming4[18];
    unsigned char ExtensionFlag;    // # of 128 byte EDID extensions
    unsigned char checksum;         // must equal zero
} PGL_DDC_EDID;

typedef struct                      /* PGL structure for PGL_loadmodelist() */
{
    int xres;
    int yres;
    int bpp;
    int depth;
    int mode_number;
    PGL_modetable modetable;
} PGL_modedef;

typedef struct                      /* PGL mode configuration structure */
{
    int           xres;                   // given x resolution
    int           yres;                   // given y resolution
    int           bpp;                    // bits per pixel
    int           depth;                  // bpp attribute (color weight, order)
    int           pitch;                  // given pitch
    int           mode_number;            // rom mode number
    int           attribute_flags;        // given attribute flags
    int           actual_xres;            // dependant on the rom mode number
    int           actual_yres;            // dependant on the rom mode number
    int           maxy;                   // maximum y line
    int           vga_aperture_status;    // VGA_APERTURE_ENABLE, VGA_APERTURE_DISABLE
    int           linear_aperture_status; // LINEAR_APERTURE_ENABLE, LINEAR_APERTURE_DISABLED
    int           linear_aperture_size;   // APERTURE_4M_ENABLE, APERTURE_8M_ENABLE
    unsigned long aperture_address;       // linear aperture address
    unsigned long vga_memreg_offset;      // engine regs in VGA aperture
    unsigned long linear_memreg_offset;   // engine regs in linear aperture
    unsigned long memory_offset;          // given video memory offset
    int           register_access_method; // USE_VGA_APERTURE, USE_LINEAR_APERTURE
    unsigned int  rom_segment;            // ROM segment of selected Mach64 card
    unsigned int  io_base_address;        // base I/O address
    int           io_base_type;           // IO_TYPE_STANDARD, IO_TYPE_FLOAT
    int           instance;               // Instance of selected Mach64 card
    int           chip_type;              // CHIP_GX_ID, CHIP_CX_ID, CHIP_CT_ID, CHIP_DT_ID
    int           chip_version;           // version of specific asic chip
    int           chip_attribute;         // CHIP_SP_STANDARD, CHIP_SP_GXE
    int           dac_attribute;          // DAC_SP_STANDARD, DAC_SP_ATI68860, DAC_SP_TVP3020, DAC_SP_IBM514
    int           fifo_check_control;     // FIFO_CHECK_NORMAL, FIFO_CHECK_TIMEOUT
    int           fifo_action_control;    // FIFO_ACTION_NONE, FIFO_ACTION_TERMINATE
    int           doubleclock_flag;       // STANDARD_CLOCK, DOUBLE_CLOCK
} PGL_modeconfig;

typedef struct                      /* PGL attribute structure */
{
    unsigned long fgcolor;
    unsigned long bgcolor;
    int           fgmix;
    int           bgmix;
    unsigned long writemask;
    int           linetype;
    int           linelength;
    unsigned long linepattern;
    int           fillsolid;
    int           fillpatternwidth;
    int           fillpatternheight;
    int           fastpolyflag;
    unsigned long fillpattern[96];
    int           srcline_y;
    int           srcrect_y;
    int           srcpoly_y;
} PGL_attribute;

typedef struct                      /* FNT font file header structure */
{
    char          reserved1;
    char          fonttype;           // 1 = BITMAP
    char          reserved2;
    long          reserved3;
    char          charwidth;
    char          charheight;
    char          reserved4;
    int           chardefsize;
    int           flags;              // bit 13: 0 = not prop, 1 = prop
    unsigned long index_table_offset; // required
    unsigned long env_table_offset;   // only for proportional
    char          firstchar;
    char          lastchar;
    unsigned long char_table_offset;  // required
    int           reserved5;
    unsigned long second_char_table_offset;
    int           reserved6;
    unsigned long third_char_table_offset;
} PGL_char_def_block;

/*
    Typical FNT font file code pages - usually five sets:

    Code page string        Language
    ----------------        --------
    "437"                   US/English
    "850"                   Multilingual
    "860"                   Portuguese
    "863"                   Canadian/French
    "865"                   Nordic

    However, all five CDB's point to the same data (US/English). Therefore,
    only the data for the first set needs to be loaded into memory.
*/
typedef struct
{
    int          code_pages;          // should be five sets
    unsigned int code_page_index;     // usually zero
    unsigned int alt_code_page_index; // not needed
    char         cp1[4];
    unsigned int cdb1_offset;         // pointer to char_def_block 1
    char         cp2[4];
    unsigned int cdb2_offset;         // pointer to char_def_block 2
    char         cp3[4];
    unsigned int cdb3_offset;         // pointer to char_def_block 3
    char         cp4[4];
    unsigned int cdb4_offset;         // pointer to char_def_block 4
    char         cp5[4];
    unsigned int cdb5_offset;         // pointer to char_def_block 5
} PGL_fnt_format;

typedef struct                      /* PGL font structure */
{
    int            charwidth;
    int            charheight;
    unsigned int   index[256];
    unsigned char  far *bit_data;
    unsigned char *unload_ptr;
} PGL_font;

typedef enum {BLUE, GREEN, RED} PGL_primarycolor;

typedef struct                      /* PGL hardware cursor structure */
{
    int           y;
    int           width;
    int           height;
    int           current_x;
    int           current_y;
    int           hot_x;
    int           hot_y;
    unsigned long cur_offset;
    unsigned long color0;
    unsigned long color1;
    unsigned int  bitmap[HWCURHEIGHT * HWCURWIDTH];
} PGL_hwcursor;

typedef struct                      /* PGL palette entry structure */
{
    int red;
    int green;
    int blue;
} PGL_palette;

typedef struct                      /* PGL coordinate structure */
{
    int x, y;
} PGL_point;

typedef struct                      /* PGL clip region structure */
{
    int left, top, right, bottom;
} PGL_scissor;

typedef struct                      /* PGL context structure */
{
    unsigned long context_mask;
    unsigned long reserved1;
    unsigned long dst_off_pitch;
    unsigned long dst_y_x;
    unsigned long dst_height_width;
    unsigned long dst_bres_err;
    unsigned long dst_bres_inc;
    unsigned long dst_bres_dec;
    unsigned long src_off_pitch;
    unsigned long src_y_x;
    unsigned long src_height1_width1;
    unsigned long src_y_x_start;
    unsigned long src_height2_width2;
    unsigned long pat_reg0;
    unsigned long pat_reg1;
    unsigned long sc_left_right;
    unsigned long sc_top_bottom;
    unsigned long dp_bkgd_clr;
    unsigned long dp_frgd_clr;
    unsigned long dp_write_mask;
    unsigned long dp_chain_mask;
    unsigned long dp_pix_width;
    unsigned long dp_mix;
    unsigned long dp_src;
    unsigned long clr_cmp_clr;
    unsigned long clr_cmp_mask;
    unsigned long clr_cmp_cntl;
    unsigned long gui_traj_cntl;
    unsigned long context_load_cntl;
    unsigned long reserved2[35];
} PGL_context;

typedef struct                      /* PGL generic colors structure */
{
    unsigned long colors_4_8[16];
    unsigned long colors_16_555[16];
    unsigned long colors_16_565[16];
    unsigned long colors_24_RGB[16];
    unsigned long colors_24_BGR[16];
    unsigned long colors_32_RGBA[16];
    unsigned long colors_32_ARGB[16];
    unsigned long colors_32_BGRA[16];
    unsigned long colors_32_ABGR[16];
} PGL_genericcolors;

/* resume word alignment */
#pragma pack()

// --------------------------------------------------------------------------
// PGL public function prototypes
// --------------------------------------------------------------------------

int              ior8(unsigned int ioindex);
unsigned int     ior16(unsigned int ioindex);
unsigned long    ior32(unsigned int ioindex);
void             iow8(unsigned int ioindex, int iodata);
void             iow16(unsigned int ioindex, unsigned int iodata);
void             iow32(unsigned int ioindex, unsigned long iodata);
void             PGL_blankscreen(void);
void             PGL_buffertoscreenblit(unsigned long huge *bufferptr,
                                        int dstleft, int dsttop,
                                        int recwidth, int recheight);
void             PGL_clearscreen(int x, int y, int width, int height);
void             PGL_close(void);
void             PGL_closeaperture(void);
void             PGL_closemode(void);
int              PGL_detectmach64(int instance);
void             PGL_disableblockwrite(void);
void             PGL_disablehwcursor(void);
void             PGL_disablelinearaperture(void);
void             PGL_disableoverscan(void);
void             PGL_disablevgaaperture(void);
void             PGL_drawline(int x1, int y1, int x2, int y2);
void             PGL_drawpixel(int x, int y);
void             PGL_drawpolyline(int n, PGL_point *vertices);
int              PGL_drawpolygon(int n, PGL_point *vertices);
void             PGL_drawrect(int x, int y, int width, int height);
void             PGL_drawto(int x, int y);
void             PGL_enableblockwrite(void);
void             PGL_enablehwcursor(void);
void             PGL_enablelinearaperture(int aperture_size);
void             PGL_enableoverscan(unsigned long color,
                                    int leftwidth, int rightwidth,
                                    int topsizewidth, int bottomwidth);
void             PGL_enablevgaaperture(void);
unsigned long    PGL_getapertureaddress(void);
PGL_attribute   *PGL_getattributeptr(void);
unsigned long    PGL_getbgcolor(void);
int              PGL_getbgmix(void);
unsigned long    PGL_getbuffersize(int recwidth, int recheight);
unsigned long    PGL_getcolorcode(int generic_color);
void             PGL_getclipsize(PGL_scissor *scissor);
PGL_crttable    *PGL_getcrttable(int table_selection);
unsigned long    PGL_getfgcolor(void);
int              PGL_getfgmix(void);
PGL_font        *PGL_getfixedfont8x8(void);
PGL_font        *PGL_getfixedfont8x14(void);
PGL_font        *PGL_getfixedfont7x15(void);
PGL_font        *PGL_getfixedfont12x20(void);
void             PGL_gethwcursorpos(PGL_point *position);
void             PGL_getmodeconfig(PGL_modeconfig *modeconfig);
PGL_modeconfig  *PGL_getmodeconfigptr(void);
int              PGL_getnpolylines(int n, PGL_point *vertices);
PGL_palette      PGL_getpalette(int index);
unsigned long    PGL_getpixel(int x, int y);
void             PGL_getposition(PGL_point *position);
unsigned long    PGL_getprimarycolor(PGL_primarycolor primarycolor,
                                     unsigned long color);
void             PGL_getqueryheader(PGL_queryheader *queryinfo);
PGL_queryheader *PGL_getqueryheaderptr(void);
int              PGL_getquerysize(int fill_type);
int              PGL_getsrcline(void);
int              PGL_getsrcpoly(void);
int              PGL_getsrcrect(void);
PGL_version      PGL_getversion(void);
unsigned long    PGL_getwritemask(void);
int              PGL_initaperture(int register_access);
void             PGL_initengine(void);
int              PGL_initmode(int x,
                              int y,
                              int bpp,
                              int depth,
                              int pitch,
                              unsigned long offset,
                              int attribute_flags,
                              int mode_number,
                              PGL_crttable far *tableinfo,
                              int eeprom_offset);
void             PGL_initpalette(void);
void             PGL_initpalettized(void);
void             PGL_loadcontext(unsigned int loadpointer,
                                 unsigned long attribute);
int              PGL_loadfont(char *fontfile, PGL_font *font);
PGL_modedef     *PGL_loadmodelist(int *modes);
unsigned long    PGL_makecolor(unsigned long r,
                               unsigned long g,
                               unsigned long b);
void             PGL_monoexpandblit(int srcleft, int srctop,
                                    int dstleft, int dsttop,
                                    int recwidth, int recheight);
void             PGL_monotoscreenblit(unsigned long huge *bufferptr,
                                      int dstleft, int dsttop,
                                      int recwidth, int recheight);
void             PGL_moveto(int x, int y);
PGL_point        PGL_ncscreentoscreenblit(int recleft, int rectop,
                                          int recwidth, int recheight,
                                          int x, int y,
                                          int saveflag);
int              PGL_queryhardware(int fill_type, void far *queryptr);
int              PGL_readblock(void huge *cpuptr,
                               unsigned long memory_offset,
                               unsigned long nbytes);
void             PGL_resetengine(void);
void             PGL_restorepalette(PGL_palette *paletteBuffer);
void             PGL_savepalette(PGL_palette *paletteBuffer);
void             PGL_screentobufferblit(unsigned long huge *bufferptr,
                                        int srcleft, int srctop,
                                        int recwidth, int recheight);
void             PGL_screentoscreenblit(int srcleft, int srctop,
                                        int dstleft, int dsttop,
                                        int recwidth, int recheight);
void             PGL_setapertureaddress(unsigned long address);
void             PGL_setbgcolor(unsigned long color);
void             PGL_setbgmix(int mix);
void             PGL_setclipsize(int left, int top, int right, int bottom);
void             PGL_setcolordepth(int depth);
void             PGL_setfgcolor(unsigned long color);
void             PGL_setfgmix(int mix);
void             PGL_setfifocontrol(int check_control, int action_control);
void             PGL_setfillpattern(int width, int height,
                                    unsigned long *fillpattern);
void             PGL_setfilltype(int type);
int              PGL_sethwcursor(int y, int width, int height,
                                 int hot_x, int hot_y,
                                 unsigned long color0, unsigned long color1,
                                 unsigned int *bitmap);
void             PGL_sethwcursorpos(int x, int y);
void             PGL_setlinepattern(unsigned long linepattern, int length);
void             PGL_setlinetype(int type);
void             PGL_setpalette(int index, PGL_palette entry);
void             PGL_setsrcline(int y);
void             PGL_setsrcpoly(int y);
void             PGL_setsrcrect(int y);
void             PGL_setwritemask(unsigned long mask);
void             PGL_text(int x, int y, char *string, PGL_font *font);
void             PGL_unblankscreen(void);
void             PGL_updateaperturestatus(void);
void             PGL_uploadcontext(unsigned int loadpointer,
                                   PGL_context far *context);
void             PGL_unloadfont(PGL_font *font);
void             PGL_unloadmodelist(PGL_modedef *modelist);
int              PGL_vgafont(int vgafontsize, PGL_font *font);
int              PGL_waitforemptyfifo(void);
int              PGL_waitforfifo(int entries);
int              PGL_waitforidle(void);
void             PGL_waitforvblank(int frames);
void             PGL_waitforvblankst(int frames);
int              PGL_writeblock(void huge *cpuptr,
                                unsigned long memory_offset,
                                unsigned long nbytes);
unsigned long    regr(unsigned int regindex);
int              regw(unsigned int regindex, unsigned long regdata);


// --------------------------------------------------------------------------
// PGL private function prototypes that are public for special usage
// --------------------------------------------------------------------------

void             pgl_blankvgascreen(void);
unsigned int     pgl_getrombase(void);
unsigned int     pgl_getvideomode(void);
unsigned long    pgl_getxyoffset(int x, int y);
void             pgl_initioreg(unsigned int iobase, int iotype);
void             pgl_initmemreg(int vga_app_status,
                                unsigned long aperture_base_address,
                                unsigned long register_memory_offset);
int              pgl_movemem(void far *src, unsigned long dest,
                             unsigned int nwords,
                             int direction);
void             pgl_setrombase(unsigned int rom_segment);
void             pgl_setvideomode(unsigned int vmode);
void             pgl_unblankvgascreen(void);
void             pgl_updateappvars(void);
int              ROM_aperturecontrol(int aperture_control);
int              ROM_fillquerystructure(int info_type, void far *ptr);
int              ROM_getiobase(PGL_iobasequery far *iobaseinfo);
int              ROM_getquerysize(int info_type);
int              ROM_loadandsetmode(int resolution_code,
                                    int pitch_code,
                                    int color_depth_code,
                                    int palettized_flag,
                                    PGL_crttable far *tableinfo,
                                    int eeprom_offset);
int              ROM_loadmodeparms(int resolution_code,
                                   int pitch_code,
                                   int color_depth_code,
                                   int palettized_flag,
                                   PGL_crttable far *tableinfo,
                                   int eeprom_offset);
int              ROM_readeeprom(unsigned int index, unsigned int far *data);
int              ROM_setdisplaymode(int display_mode,
                                    int palettized_flag,
                                    int far *doubleclock_flag);
int              ROM_setdpmscontrol(int service_code);
int              ROM_shortquery(PGL_shortquery far *shortqueryinfo);
int              ROM_DDCquery(unsigned int far *rom_support,
                              unsigned int far *monitor_support);
int              ROM_DDCreadEDID(unsigned int bufsize,
                                 PGL_DDC_EDID far *packet);
int              ROM_writeeeprom(unsigned int index, unsigned int data);

#endif

