/*==========================================================================
* CFGAPP.C - Copyright (c) 1994 ATI Technologies Inc. All rights reserved  *
*                                                                          *
* PGL functions to enable and disable the apertures (VGA or Linear). This  *
* affect the memory mapped routines REGR() and REGW().                     *
* ======================================================================== */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <conio.h>
#include <memory.h>
#include <dos.h>
#include <malloc.h>

#include "..\inc\atim64.h"
#include "..\inc\pgl.h"
#include "..\inc\pglglob.h"

/* --------------------------------------------------------------------------
  pgl_updateappvars - update aperture variables in MEMREG.ASM module

  Since REGW() and REGR() depends on these variables, they must be updated
  any time the aperture status changes (VGA or LINEAR).
-------------------------------------------------------------------------- */
void pgl_updateappvars(void)
{
    if (PGL_modecfg.vga_aperture_status == VGA_APERTURE_ENABLED)
    {
        // update static variables in MEMREG.ASM module
        pgl_initmemreg(1, PGL_modecfg.aperture_address,
                       PGL_modecfg.linear_memreg_offset);
    }
    else
    {
        // update static variables in MEMREG.ASM module
        pgl_initmemreg(0, PGL_modecfg.aperture_address,
                       PGL_modecfg.linear_memreg_offset);
    }
}

/* --------------------------------------------------------------------------
  PGL_enablevgaaperture - enable the vga aperture
  
  This function enables the VGA aperture and sets the PGL_modeconfig
  structure contents accordingly. The PGL functions will use the new
  settings.
-------------------------------------------------------------------------- */
void PGL_enablevgaaperture(void)
{
    iow32(ioCONFIG_CNTL, ior32(ioCONFIG_CNTL) | VGA_APERTURE_ENABLE);
    PGL_modecfg.vga_aperture_status = VGA_APERTURE_ENABLED;

    // update memory mapped register dependent variables
    pgl_updateappvars();
}

/* --------------------------------------------------------------------------
  PGL_disablevgaaperture - disable the vga aperture
  
  This function disables the VGA aperture and sets the PGL_modeconfig
  structure contents accordingly. The PGL functions will use the new
  settings. It is important to note that if both the linear aperture and
  the vga aperture are disabled, the PGL functions will fail to operate
  correctly. It is the application's responsibility to insure that this
  does not occur.
-------------------------------------------------------------------------- */
void PGL_disablevgaaperture(void)
{
    iow32(ioCONFIG_CNTL, ior32(ioCONFIG_CNTL) & (~VGA_APERTURE_ENABLE));
    PGL_modecfg.vga_aperture_status = VGA_APERTURE_DISABLED;

    // update memory mapped register dependent variables
    pgl_updateappvars();
}

/* --------------------------------------------------------------------------
  PGL_enablelinearaperture - enable the linear aperture
  
  This function enables the linear aperture for 4M or 8M. For the Mach64 CT
  and DT variants, the linear aperture size must be 8M. The linear aperture
  address is set by the ROM at boot time. The address is assumed to be
  correct. The PGL functions will use the new settings.
-------------------------------------------------------------------------- */
void PGL_enablelinearaperture(int aperture_size)
{
    // no 4Meg size exists for Mach64 CT & DT variants so re-assign size
    if ((PGL_modecfg.chip_type == CHIP_CT_ID) ||
        (PGL_modecfg.chip_type == CHIP_DT_ID))
    {
        if (aperture_size == APERTURE_4M_ENABLE)
        {
            aperture_size = APERTURE_8M_ENABLE;
        }
    }

    // enable linear aperture with given size
    if (aperture_size == APERTURE_4M_ENABLE) // 4M size
    {
        iow32(ioCONFIG_CNTL, (ior32(ioCONFIG_CNTL) & 0xfffffffc) | APERTURE_4M_ENABLE);
        PGL_modecfg.linear_memreg_offset = 0x3ffc00;
        PGL_modecfg.linear_aperture_size = APERTURE_4M_ENABLE;
    }
    else                                     // 8M size
    {
        iow32(ioCONFIG_CNTL, (ior32(ioCONFIG_CNTL) & 0xfffffffc) | APERTURE_8M_ENABLE);
        PGL_modecfg.linear_memreg_offset = 0x7ffc00;
        PGL_modecfg.linear_aperture_size = APERTURE_8M_ENABLE;
    }
    PGL_modecfg.linear_aperture_status = LINEAR_APERTURE_ENABLED;

    // update memory mapped register dependent variables
    pgl_updateappvars();
}

/* --------------------------------------------------------------------------
  PGL_disablelinearaperture - disable the linear aperture
  
  This function disables the linear aperture. It is important to note that
  if both the linear aperture and the vga aperture are disabled, the PGL
  functions will fail to operate correctly. It is the application's
  responsibility to insure that this does not occur.
-------------------------------------------------------------------------- */
void PGL_disablelinearaperture(void)
{
    iow32(ioCONFIG_CNTL,
          ior32(ioCONFIG_CNTL) & (~(APERTURE_4M_ENABLE | APERTURE_8M_ENABLE)));
    PGL_modecfg.linear_aperture_status = LINEAR_APERTURE_DISABLED;

    // update memory mapped register dependent variables
    pgl_updateappvars();
}

