/*==========================================================================
* DRAWLINE.C - Copyright (c) 1994 ATI Technologies Inc. All rights reserved*
*                                                                          *
* PGL functions to draw lines.                                             *
* ======================================================================== */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <conio.h>

#include "..\inc\atim64.h"
#include "..\inc\pgl.h"
#include "..\inc\pglglob.h"

/* --------------------------------------------------------------------------
  PGL_moveto - change current position

  The current drawing position is set to (x, y).
-------------------------------------------------------------------------- */
void PGL_moveto(int x, int y)
{
    PGL_waitforfifo(1);
    if (PGL_modecfg.bpp == 24)
    {
        x = x * 3;
    }
    regw(DST_Y_X, ((unsigned long)(x) << 16) | y);
}

/* --------------------------------------------------------------------------
  pgl_drawln - draw a solid one pixel thick line

  The drawing of the last pixel in the line is determined by the current
  setting of the DST_CNTL register.
-------------------------------------------------------------------------- */
void pgl_drawln(int x1, int y1, int x2, int y2)
{
    int dx, dy;
    int small, large;
    int x_dir, y_dir, y_major;
    unsigned long err, inc, dec, temp;

    // use special routine for 24 bpp modes
    if (PGL_modecfg.bpp == 24)
    {
        if (x1 == x2)
        {
            // vertical 24 bpp lines
            pgl_vline24(y1, y2, x1);
        }
        else if (y1 == y2)
        {
            // horizontal 24 bpp lines
            pgl_hline24(x1, x2, y1);
        }
        else
        {
            // bressenham 24 bpp lines
            pgl_bressline24(x1, y1, x2, y2);
        }
        PGL_waitforidle();
        PGL_moveto(x2, y2);
        return;
    }

    /* determine x & y deltas and x & y direction bits */
    if (x1 < x2)
    {
        dx = x2 - x1;
        x_dir = 1;
    }
    else
    {
        dx = x1 - x2;
        x_dir = 0;
    }

    if (y1 < y2)
    {
        dy = y2 - y1;
        y_dir = 2;
    }
    else
    {
        dy = y1 - y2;
        y_dir = 0;
    }

    /* determine x & y min and max values; also determine y major bit */
    if (dx < dy)
    {
        small = dx;
        large = dy;
        y_major = 4;
    }
    else
    {
        small = dy;
        large = dx;
        y_major = 0;
    }

    /* calculate bressenham parameters and draw line */
    err = (unsigned long)((2 * small) - large);
    inc = (unsigned long)(2 * small);
    dec = 0x3ffff - ((unsigned long)(2 * (large - small)));


    // save vital regs
    PGL_waitforidle();
    temp = regr(DST_CNTL);

    // draw bressenham line
    regw(DST_Y_X, ((unsigned long)(x1) << 16) | y1);
    regw(DST_CNTL, (temp & 0x60) | y_major | y_dir | x_dir |
                   DST_Y_TILE | DST_X_TILE);
    regw(DST_BRES_ERR, err);
    regw(DST_BRES_INC, inc);
    regw(DST_BRES_DEC, dec);
    regw(DST_BRES_LNTH, (unsigned long)(large + 1));

    // restore
    regw(DST_CNTL, temp);
}

/* --------------------------------------------------------------------------
  PGL_drawline - draw a line from (x1, y1) to (x2, y2)

  A Bresenham one pixel thick line is drawn from (x1, y1) to (x2, y2)
  using the current foreground color, mix, and line pattern. The drawing of
  the last pixel in the line is determined by the current setting of the
  DST_CNTL register.
-------------------------------------------------------------------------- */
void PGL_drawline(int x1, int y1, int x2, int y2)
{
    unsigned long temp1, temp2, temp3, temp4, temp5, temp6, temp7;
    unsigned long offset;

    if (PGL_attr.linetype == SOLID_LINE)
    {
        // draw bressenham line
        pgl_drawln(x1, y1, x2, y2);
    }
    else
    {
        // calculate qword offset to start of off-screen memory
        offset = pgl_getxyoffset(0, PGL_attr.srcline_y) / 8;

        // save vital regs
        PGL_waitforidle();
        temp1 = regr(DST_OFF_PITCH);
        temp2 = regr(SRC_OFF_PITCH);
        temp3 = regr(DP_PIX_WIDTH);
        temp4 = regr(DP_SRC);
        temp5 = regr(SRC_CNTL);
        temp6 = regr(DP_MIX);

        // place patterned source line in off-screen memory
        regw(DST_OFF_PITCH, (temp1 & 0xffc00000) | offset);
        if (PGL_modecfg.bpp == 4)
        {
            regw(DP_PIX_WIDTH, 0);
        }
        else
        {
            regw(DP_PIX_WIDTH, BYTE_ORDER_LSB_TO_MSB);
        }
        regw(DP_SRC, MONO_SRC_HOST | FRGD_SRC_FRGD_CLR);
        regw(DP_MIX, FRGD_MIX_ONE | BKGD_MIX_ZERO);
        regw(DST_Y_X, 0);
        regw(DST_HEIGHT_WIDTH, 0x00200001);  // 32 x 1
        regw(HOST_DATA0, PGL_attr.linepattern);

        // setup engine to use source in blit mode
        PGL_waitforidle();
        regw(DST_OFF_PITCH, temp1);
        regw(SRC_OFF_PITCH, (temp2 & 0xffc00000) | offset);
        if (PGL_modecfg.bpp == 4)
        {
            regw(DP_PIX_WIDTH, temp3 & 0x000000ff);
        }
        else
        {
            regw(DP_PIX_WIDTH, (temp3 & 0x000000ff) | BYTE_ORDER_LSB_TO_MSB);
        }
        regw(SRC_Y_X, 0);
        regw(SRC_CNTL, SRC_LINE_X_LEFT_TO_RIGHT | SRC_PATTERN_ENABLE);
        regw(SRC_HEIGHT1_WIDTH1, ((unsigned long)(PGL_attr.linelength) << 16) | 1);
        regw(DP_SRC, MONO_SRC_BLIT | FRGD_SRC_FRGD_CLR);
        regw(DP_MIX, temp6);

        // draw bressenham line
        if (PGL_attr.linetype == INVISIBLE_LINE)
        {
            PGL_waitforidle();
            temp7 = regr(DP_WRITE_MASK);
            regw(DP_WRITE_MASK, 0);
            pgl_drawln(x1, y1, x2, y2);
            regw(DP_WRITE_MASK, temp7);
        }
        else
        {
            pgl_drawln(x1, y1, x2, y2);
        }

        // restore
        PGL_waitforfifo(5);
        regw(DST_OFF_PITCH, temp1);
        regw(SRC_OFF_PITCH, temp2);
        regw(DP_PIX_WIDTH, temp3);
        regw(DP_SRC, temp4);
        regw(SRC_CNTL, temp5);
    }
}

/* --------------------------------------------------------------------------
  PGL_drawto - draw a line from current position to (x, y)

  A one pixel thick Bresenham line is drawn from the current position to
  (x, y) using the current foreground color, mix, and line pattern.
  The drawing of the last pixel in the line is determined by the current
  setting of the DST_CNTL register.
-------------------------------------------------------------------------- */
void PGL_drawto(int x, int y)
{
    PGL_waitforidle();
    if (PGL_modecfg.bpp == 24)
    {
        PGL_drawline((int)(regr(DST_X)/3), (int)regr(DST_Y), x, y);
    }
    else
    {
        PGL_drawline((int)regr(DST_X), (int)regr(DST_Y), x, y);
    }
}

/* --------------------------------------------------------------------------
  PGL_drawpolyline - draw multiple line segments

  Multiple one pixel thick Bresenham lines are drawn from a list of "n" lines
  using the current array of points referenced by "vertices". Note that the
  resulting outline is closed only if the first point is the same as the last.
  The current foreground color, mix, and line pattern are used. The drawing
  of the last pixel in the line is determined by the current setting of the
  DST_CNTL register.
-------------------------------------------------------------------------- */
void PGL_drawpolyline(int n, PGL_point *vertices)
{
    register int i;

    if (n < 1) return;

    for (i = 0; i < n; i++)
    {
        PGL_drawline(vertices[i].x, vertices[i].y, vertices[i+1].x, vertices[i+1].y);
    }
}

/* --------------------------------------------------------------------------
  PGL_drawpixel - draw a pixel at (x, y)

  A pixel is drawn at coordinates (x, y) using the current foreground color,
  and mix. This is essentially a rectangle fill with a width & height of 1.
-------------------------------------------------------------------------- */
void PGL_drawpixel(int x, int y)
{
    unsigned long rotation, temp1, temp2;

    // save vital regs
    PGL_waitforidle();
    temp1 = regr(DP_SRC);
    temp2 = regr(DST_CNTL);

    // set pixel color source to foreground color register
    regw(DP_SRC, FRGD_SRC_FRGD_CLR);

    // draw pixel
    if (PGL_modecfg.bpp == 24)
    {
        rotation = pgl_get24bpprotation(x);
        regw(DST_CNTL, DST_24_ROTATION_ENABLE | rotation |
                       DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT);
        regw(DST_Y_X, ((unsigned long)(x * 3) << 16) | y);
        regw(DST_HEIGHT_WIDTH, 0x00030001);      // 3 x 1
    }
    else
    {
        regw(DST_CNTL, DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT |
                       DST_Y_TILE | DST_X_TILE);
        regw(DST_Y_X, ((unsigned long)(x) << 16) | y);
        regw(DST_HEIGHT_WIDTH, 0x00010001);      // 1 x 1
    }

    // restore
    regw(DP_SRC, temp1);
    regw(DST_CNTL, temp2);
}

/* --------------------------------------------------------------------------
  pgl_bressline24 - draw a bressenham line from (x1, y1) to (x2, y2).

  A line is drawn using the current foreground color and mix using an
  emulated bressenham line routine. This routine is called for 24 bpp
  modes.
-------------------------------------------------------------------------- */
void pgl_bressline24(int x1, int y1, int x2, int y2)
{
    int x, y, xend, yend, dx, dy;
    int d, incr1, incr2, incr3;
    unsigned long rotation, temp1, temp2;

    PGL_waitforidle();
    temp1 = regr(DST_CNTL);
    temp2 = DST_24_ROTATION_ENABLE | DST_LAST_PEL |
            DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT;

    // ------------------ Bressenham line routine ----------------------

    dx = abs(x2 - x1);
    dy = abs(y2 - y1);

    // check slope
    if (dy <= dx) // slope <= 1

    {
        if (x1 > x2)
        {
            x = x2;
            y = y2;
            xend = x1;
            dy = y1 - y2;
        }
        else
        {
            x = x1;
            y = y1;
            xend = x2;
            dy = y2 - y1;
        }

        d = (2 * dy) - dx;
        incr1 = 2 * dy;
        incr2 = 2 * (dy - dx);
        incr3 = 2 * (dy + dx);

        regw(DST_HEIGHT, 1);
        regw(DST_Y, y);

        do
        {
            PGL_waitforfifo(4);
            rotation = pgl_get24bpprotation(x);
            regw(DST_CNTL, temp2 | rotation);
            regw(DST_X, x * 3);
            regw(DST_WIDTH, 3);

            x++;

            if (d >= 0)
            {
                if (dy <= 0)
                {
                    d = d + incr1;
                }
                else
                {
                    y++;
                    regw(DST_Y, y);
                    d = d + incr2;
                }
            }
            else
            {
                if (dy >= 0)
                {
                    d = d + incr1;
                }
                else
                {
                    y--;
                    regw(DST_Y, y);
                    d = d + incr3;
                }
            }
        } while (x <= xend);
    }
    else          // slope > 1
    {
        if (y1 > y2)
        {
            y = y2;
            x = x2;
            yend = y1;
            dx = x1 - x2;
        }
        else
        {
            y = y1;
            x = x1;
            yend = y2;
            dx = x2 - x1;
        }

        d = (2 * dx) - dy;
        incr1 = 2 * dx;
        incr2 = 2 * (dx - dy);
        incr3 = 2 * (dx + dy);

        regw(DST_HEIGHT, 1);

        do
        {
            PGL_waitforfifo(3);
            rotation = pgl_get24bpprotation(x);
            regw(DST_CNTL, temp2 | rotation);
            regw(DST_Y_X, ((unsigned long)(x * 3) << 16) | y);
            regw(DST_WIDTH, 3);

            y++;

            if (d >= 0)
            {
                if (dx <= 0)
                {
                    d = d + incr1;
                }
                else
                {
                    x++;
                    d = d + incr2;
                }
            }
            else
            {
                if (dx >= 0)
                {
                    d = d + incr1;
                }
                else
                {
                    x--;
                    d = d + incr3;
                }
            }
        } while (y <= yend);
    }

    PGL_waitforfifo(1);
    regw(DST_CNTL, temp1);
    PGL_moveto(x2, y2);
}

/* --------------------------------------------------------------------------
  pgl_vline24 - draw a vertical line from (x, y1) to (x, y2).

  A vertical line is drawn using the current foreground color and mix. This
  routine is called for 24 bpp modes.
-------------------------------------------------------------------------- */
void pgl_vline24(int y1, int y2, int x)
{
    unsigned long temp, rotation;
    int swap;

    // save vital regs
    PGL_waitforidle();
    temp = regr(DST_CNTL);

    // insure height is positive
    if (y1 > y2)
    {
        swap = y1;
        y1 = y2;
        y2 = swap;
    }

    // set 24 bpp rotation value
    rotation = pgl_get24bpprotation(x);

    regw(DST_Y_X, ((unsigned long)(x * 3) << 16) | y1);
    regw(DST_CNTL, (temp & 0x80) | rotation |
                   DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT | DST_Y_TILE);
    regw(DST_HEIGHT_WIDTH, 0x00030000 | (y2 - y1 + 1));     // width = 3

    // restore
    regw(DST_CNTL, temp);
}

/* --------------------------------------------------------------------------
  pgl_hline24 - draw a horizontal line from (x1, y) to (x2, y).

  A horizontal line is drawn using the current foreground color and mix. This
  routine is called for 24 bpp modes.
-------------------------------------------------------------------------- */
void pgl_hline24(int x1, int x2, int y)
{
    unsigned long temp, rotation;
    int swap;

    // save vital regs
    PGL_waitforidle();
    temp = regr(DST_CNTL);

    // insure width is positive
    if (x1 > x2)
    {
        swap = x1;
        x1 = x2;
        x2 = swap;
    }

    // set 24 bpp rotation value
    rotation = pgl_get24bpprotation(x1);

    regw(DST_Y_X, ((unsigned long)(x1 * 3) << 16) | y);
    regw(DST_CNTL, (temp & 0x80) | rotation |
                   DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT | DST_X_TILE);
    regw(DST_HEIGHT_WIDTH, ((unsigned long)((x2 - x1 + 1) * 3) << 16) | 1);

    // restore
    regw(DST_CNTL, temp);
}

