

;
; Pixel block I/O functions.
;
; Entries -	SAI
;		DAI
;		WHR
;		RHR
;		WHS
;		RHS
;		RHC
;		WHC
;		WDA
;		RDA
;		RRD
;		WRD
;		CAI
;		SUC
;
; Interpreter vars :
; 		aixorg, aiyorg,	; x,y coord of aoi origin.
; 		aimjsz, aimnsz	; major, minor aoi size.
; 		aiflag		; capctl image for aoi processing.


sai:
	php
	rep	#0x30
	
	jsl	>0,Getb
	and	##0xff
	beq	$1
	lda	##4		; y major aoi.
$1:	sta	<aiflag

	jsl	>0,Gcoor
	stx	<aixorg
	sty	<aiyorg
	bra	dai1

dai:
	php
	rep	#0x30

	lda	<xpos
	sta	<aixorg
	lda	<ypos
	sta	<aiyorg
	stz	<aiflag
dai1:
	jsl	>0,Gcoor

	sec
	txa
	sbc	<aixorg
	tax
	bpl	$1
	eor	##-1
	inc	a
	tax
	lda	##2
	tsb	<aiflag
$1:
	sec
	tya
	sbc	<aiyorg
	tay
	bpl	$2
	eor	##-1
	inc	a
	tay
	inc	<aiflag
$2:
	lda	<aiflag
	bit	##4
	beq	$3
	lsr	a
	bcs	$21
	and	##0xffff-2
$21:	ora	##4
	sta	<aiflag
	txa
	tyx
	tay
$3:
	inx
	iny
	stx	<aimjsz
	sty	<aimnsz

	plp
	rtl

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Local routine to set up for aoi scanning.		;
; Set cap to the aoi origin.  Set up temp, temp1 so 	;
; that storing <temp at (<temp1) sets the major cap	;
; to its original value.				;
; Point	temp1 at Xcap/Ycap for xmajor/ymajor aoi,	;
; set	temp = aixorg/aiyorg for xmajor/ymajor aoi.	;
; Set capctl.						;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

aoiset:
	php
	rep	#0x30
;
; Set cap to aoi origin.
;
	lda	<aixorg
$wdpu:	bit	Dpdone-1	; wait for dpu done.
	bvc	$wdpu
	sta	Xcap
	lda	<aiyorg
	sta	Ycap
;
; Set cap direction and major/minor bits.
; 
	ldx	##Xcap		; assume x major.
	ldy	<aixorg		; assumed original value for major cap.
	lda	Capctl
	and	##127-7		; clear direction and major/minor bits.
	ora	<aiflag		; or in the aoi dir major/minor bits.
	sta	Capctl
	bit	##4		; did we select y major ?
	beq	$1		; br if no.
	inx			; yes, point x at Ycap.
	inx
	ldy	<aiyorg		; get original value for y major cap.
$1:
	stx	<temp1		; point temp1 at major cap and
	sty	<temp		; save original value for major cap.

	plp
	rts

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; RHS - Read Horizontal Scan.			;
; Send all the pixels in the aoi to the host,	;
; in the order indicated by the aoi definition.	;
; (Horizontal is an historical misnomer).	;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

rhs:
	php
	rep	#0x30
	bsl	aoiset		; set up for aoi scanning.
	ldy	<aimnsz		; get # lines in aoi.
$nxtlin:
	rep	#0x20
	lda	<temp		; get aoi major cap origin.
	sta	(<temp1)	; put at major cap.
	ldx	<aimjsz		; get # lines in aoi.
	sep	#0x20
$loop:	
	lda	Vmimaj		; get pixel, advance cap to next.
	jsl	>0,Sendb	; send the pixel.
	dex			; reach end of line ?
	bne	$loop		; loop if no.

	bit	Vmimin		; advance minor cap to next line.
	dey			; more lines ?
	bne	$nxtlin		; go do next if yes.

	plp
	rtl

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; WHS - Write Horizontal Scan		;
; Write the entire aoi with pixels	;
; supplied by the host.			;
; (Horizontal is an historical misnomer).;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

whs:
	php
	rep	#0x30

	bsl	aoiset
	ldy	<aimnsz
$nxtlin:
	rep	#0x20
	lda	<temp		; get major org.
	sta	(<temp1)	; put at x or y cap.

	ldx	<aimjsz
	sep	#0x20
$loop:	
	jsl	>0,Getb

;	sta	Vmimaj

	sta	Clpmaj
	dex
	bne	$loop

	bit	Vmimin
	dey
	bne	$nxtlin

	plp
	rtl

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; WHC - write count pixels,	;
; starting at CAP.  Advance CAP	;
; in direction indicated by SUC	;
; command.			;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

whc:
	php
	rep	#0x30
	bsl	sucset		; setup for non-aoi pixel transfer.
	jsl	>0,Getw		; get pixel count.
	tax
$loop:	jsl	>0,Getb		; get pixel value from host.
	sta	DaWda		; write to video and advance cap.
	dex
	bne	$loop
	bsl	setend
	plp
	rtl

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; RHC - send count pixels to host.;
; Reads like WHC writes.	;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

rhc:
	php
	rep	#0x30
	bsl	sucset		; setup for non-aoi pixel transfer.
	jsl	>0,Getw		; get pixel count.
	tax
$loop:	lda	DaRda		; read pixel, advance cap.
	jsl	>0,Sendb	; send pixel value to host.
	dex
	bne	$loop
	bsl	setend
	plp
	rtl


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; RRD - DMA version of rhc.	;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

rrd:
	php
	bsl	vdmset		; setup for video dma.
	bcs	$error
	bsl	sucset		; setup for non-aoi pixel transfer.
	bsl	vdmard		; do video to host dma.
	bsl	setend
$error	plp
	rtl

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; WRD - dma version of whc	;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

wrd:
	php
	bsl	vdmset		; setup for video dma.
	bcs	$error
	bsl	sucset		; setup for non-aoi pixel transfer.
	bsl	vdmawr		; do host to video dma.
	bsl	setend
$error	plp
	rtl

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; RDA - DMA version of rhs.	;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

rda:
	php
	bsl	vdmset		; setup for video dma.
	bcs	$error
	bsl	aoiscn		; setup for dpu aoi scanning.
	bsl	vdmard		; do video to host dma.
$error	plp
	rtl

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; WDA - DMA version of whs	;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

wda:
	php
	bsl	vdmset		; setup for video dma.
	bcs	$error
	bsl	aoiscn		; setup for dpu aoi scanning.
	bsl	vdmawr		; do host to video dma.
$error	plp
	rtl


;
; Video dma setup routine.
; Returns carry set on error
; (local mode, or no parallel 
; interface).  Called by 
; wda, rda, wrd, rrd.
;
vdmset:	
	clc
	php
	sep	#0x20
	bit	<locsw		; no DMA in local mode !
	bmi	$error
	bit	0xff1e		; parallel interface present ?
	bvs	$error		; don't bother if no.
	lda	#1
	jsl	>0,Cdmctl	; enable dma.	
	plp
	rts
$error	plp
	sec
	rts

;
; Setup for drawing accelerator to
; do aoi scanning (read/write pixel
; and advance cap to 'next' pixel.
; Called by rda, wda, whs, rhs
;

aoiscn:
	php
	rep	#0x30
	bsl	aoiset		; set up for aoi scanning.
	lda	<aimnsz		; get # lines in aoi.
	dec	a
	sta	0xfe02
	lda	<aimjsz		; ditto # pixels/line.
	dec	a
	sta	0xfe00
	sta	0xfe06
	lda	<temp		; get major origin.
	sta	0xfe04
	plp
	rts

;
; SUC - setup counters for
; non-aoi pixel transfers
; (rhc, whc, rrd, wrd).
; Bit 0 = 1 for x major transfer,
; bit 2 = 1 for x up,
; bit 3 = 1 for y up.
; suc converts this bit assignment to
; that appropriate for use in the Capctl
; register, result in <sucbyt.
; Note - default value for sucbyt is 1
; (x major up, y down).

suc:
	php
	sep	#0x30
	jsl	>0,Getb
	ldx	#0		; capctl assuming x major
	ldy	#4		; ditto y major.
;
; if x cap direction is negative,
; set bit 0 in x reg, bit 1 in y reg.
;
	bit	#4		; x up ?
	bne	$xup		; br if yes.
	inx			; no, set major down in x.
	inx
	iny			; set minor down in y.
;
; if y cap direction is negative,
; set bit 1 in x reg, bit 0 in y reg.
;
$xup:	bit	#8		; y up ?
	bne	$yup		; br if yes.
	inx			; no, set minor down in x.
	iny
	iny			; set major down in y.
;
; if x is major cap, x reg has desired
; sucbyt, else it's in y reg.
;
$yup:	bit	#1		; x major ?
	bne	$done		; br if yes, x reg has all the bits.
	tyx
$done	stx	<sucbyt
	plp
	rtl

;
; Setup routine for whc,rhc,wrd,rrd.
; These routines are similar to wda
; and rda, but don't use aoi. The
; suc command determines the direction
; of transfer (x/y major, x/y up down)
; and the area is the entire window.
; Also the transfer starts at CAP, which isn't 
; necessarily at the corner of anything, so the
; first line/column length is a special case.
; This routine and aoiset should be consolidated - 
; they do very similar things but with different variables.
; Also it might then be possible to use the drawing 
; accelerator for the non-dma aoi routines.
;

sucset:	
	php
	rep	#0x30

	ldx	<xpos
	ldy	<ypos
$1:	bit	Dpdone-1
	bvc	$1
	stx	Xcap
	sty	Ycap

	ldx	Rbx		; save before alterations
	ldy	Rby

	lda	<winox		; clipping disabled, make virtual the
	sta	Xorg		; same as physical
	lda	<winoy
	sta	Yorg
	lda	##1280
	sta	<windx
	lda	##1024
	sta	<windy

	stx	Rbx
	sty	Rby

	ldx	<windx
	ldy	<windy

;	lda	<windy		; remaining lines in window = 
;	sec			; top(windy+worgy) - ypos
;	sbc	<ypos
;	clc
;	adc	<worgy
;	tay

	lda	##4
	bit	<sucbyt
	bne	$ymaj

	sty	0xfe02		; minor count (#lines).
	stx	0xfe06		; count for all lines except first.
;
; count for first line is # pixels to next edge,
; windx-xpos if x up, else xpos if x down.
; major cap reload value is 0 if x up, windx if 
; x down.
;
	lda	##2
	bit	<sucbyt		; going left or right ?

	bne	$left		; br to go left
	lda	##0		; left edge of screen
	sta	0xfe04		; major cap reload value is at left.
	lda	<windx		; count for 1st line= right edge
	clc
	adc	<worgx
	sec
	sbc	<xpos		; (windx+worgx) - xpos
	bra	$x1
$left
	lda	<windx
	sta	0xfe04
	lda	Rbx
$x1	sta	0xfe00		; count for first line
	bra	$2	

$ymaj
	stx	0xfe02		; number columns.
	sty	0xfe06		; number pixels/column except first.
;
; count for first column is # pixels to next edge, which is
; windy+worgy-ypos if y up, else ypos if y down.
;
	lda	##2
	bit	<sucbyt		; going up or down ?
	bne	$down
	ldx	##0
	stx	0xfe04		; major cap reload is at bottom.
	lda	<windy
	clc
	adc	<worgy
	sec
	sbc	<ypos
	bra	$y1
$down
	lda	<windy		; major cap reload value
	sta	0xfe04		; is at top.
	lda	Rby
$y1	sta	0xfe00		; count for 1st line
$2
	sep	#0x20
	lda	Capctl
	and	#127-7
	ora	<sucbyt
	sta	Capctl		
	
	plp
	rts

setend:
	php
	rep	#0x30
	lda	<worgx
	sta	Xorg
	lda	<worgy
	sta	Yorg
	lda	Xwtop
	sta	<windx
	lda	Ywtop
	sta	<windy
	plp
	rts

;
; Video to host dma.
; Called from RDA,RRD commands.
;
vdmard:
	php
	sep	#0x20
	phd
	pea	##0xff00
	pld
$loop:
	lda	DaRda		; read 2 pixels - they get latched
	lda	DaRda		; in host interface register.

	sta	<DmaEna-0xff00	; request DMA cycle.
$wait:	bit	<PolDev-0xff00	; test data flag and command out.
	bvs	$loop		; br if DMA cycle complete.
	bpl	$wait		; wait unless command out from host.
$done:
	pld
	lda	#0
	jsl	>0,Cdmctl	; turn off DMA.
	sta	DevEnd		; device end interrupt to host.
	plp
	rts
;
; Host to video dma.
; Called from WDA, WRD commands.
;

vdmawr:
	php
	sep	#0x30
	sta	DmaEna		; request DMA cycle.
	phd
	pea	##0xff00
	pld
	bra	$wait		; go wait for data.
$gotit:
	lda	<CdmPrt-0xff00	; read first pixel.
	ldx	<CdmPrt+1-0xff00 ; read next pixel.
	sta	<DmaEna-0xff00	; request DMA cycle.
	sta	DaWda		; write to video.
	stx	DaWda		; write next pixel.
$wait:
	bit	<PolDev-0xff00	; test data flag and command out.
	bvs	$gotit		; br if cycle complete, go read data.
	bpl	$wait		; wait unless command out.
$done:
	pld
	lda	#0
	jsl	>0,Cdmctl	; turn off DMA.
	sta	DevEnd		; device end interrupt to host.
	plp
	rts

	

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; RHR - Read Horizontal Runs.		;
; Send the aoi contents to the host	;
; in run-length encoded form		;
; (Horizontal is an historical misnomer).;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

rhr:
$count	equ	temp2
$last	equ	temp2+2

	php
	bsl	aoiset
	sep	#0x20
	rep	#0x10

	ldy	<aimnsz		; # lines to do.
	ldx	<aimjsz		; pixels/line.
$newrun:
	stz	<$count		; reset run length.
	lda	Vm		; get color current run.
	sta	<$last		; save it.
$loop:
	lda	<$last		; get run color.
	cmp	Vm		; this pixel part of run ?
	bne	$endrun		; br if not, output run.
	bit	Vmimaj		; bump cap to next pixel.
	inc	<$count		; inc run length.
	dex			; more pixels this line ?
	bne	$nxtpix		; br if yes.
	rep	#0x20		; no, set cap to beginning of line.
	lda	<temp
	sta	(<temp1)
	sep	#0x20
	ldx	<aimjsz		; get # pixels/line.
	bit	Vmimin		; bump to next line.
	dey			; more lines ?
	beq	$endrun		; br if no.
$nxtpix:
	lda	<$count
	cmp	#0xff		; run length at max ?
	bne	$loop		; br if no.
$endrun:
	lda	<$count
	jsl	>0,Sendb
	lda	<$last
	jsl	>0,Sendb
	cpy	##0
	bne	$newrun
$done:
	lda	#0
	jsl	>0,Sendb
	plp
	rtl	

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; WHR - Write Horizontal Runs.	;
; Write the aoi with run-length	;
; encoded data from host.	;
; (Horizontal is an historical	;
; misnomer).			;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; Note - this routine uses the soon to be
; obsolete DaWnp functionand should be modified
; to do DaDvr instead.

whr:

$count	equ	temp2		; run length.
$remain	equ	temp2+2		; # pixels left this line.

	php
	rep	#0x30
	bsl	aoiset

	ldx	<aimjsz		; remainder this line in x always.
	ldy	<aimnsz		; # lines in y always.
$loop:
	jsl	>0,Getb		; get run length.
	and	##0xff		; done if 0.
	beq	$done
	sta	<$count		; save run length.
	jsl	>0,Getb		; get run color.
$w0:	bit	Dpdone-1	; wait dpu done.
	bvc	$w0
	sta	DaColr		; set dpu color.
$loop1:
	cpx	<$count		; will run fit on current line ?
	bcc	$ovf		; br if too big
	beq	$ovf		; or if fills it exactly.
	lda	<$count
$w1:	bit	Dpdone-1	; wait for dpu done.
	bvc	$w1
	sta	0xfe00		; write count to da.
	sta	DaWnp		; write count pixels.
	sec
	txa			; reduce reminder by count.
	sbc	<$count
	tax
	bra	$loop		; go get next count/color.
;
; run won't fit on current line.
; write rest of line and reduce
; count by remain, and move cap to 
; next line.
;
$ovf:
	bit	Dpdone-1	; wait for dpu done.
	bvc	$ovf
	stx	0xfe00
	sta	DaWnp		; write rest of line.

	stx	<$remain	; reset remainder size 
	ldx	<aimjsz		; to length of line.

	lda	<temp		; get starting value for major cap.
$w2:	bit	Dpdone-1	; wait for dpu done
	bvc	$w2
	sta	(<temp1)	; set major cap to beginning of line.

	lda	<aiflag		;;; kludge due to obsolete
	tsb	Capctl		;;; DaWnp function.

	bit	Vmimin		; advance minor cap to next line.
	sec			; reduce count by amount
	lda	<$count		; just written.
	sbc	<$remain
	sta	<$count
	beq	$1
	dey			; more lines ?
	bne	$loop1		; yep, go do it.
	bra	$gobble
$1	dey
	bne	$loop
;
; aoi filled up, just gobble 
; rest of the input.
;

$gobble:
	jsl	>0,Getb		; count
	beq	$done
	jsl	>0,Getb		; color
	bra	$gobble
$done:
	bit	Dpdone-1	; wait for dpu done.
	bvc	$done
	lda	<color		; set da color back to that
	sta	DaColr		; assumed by firmware.
	plp
	rtl
			

cai:
	php
	rep	#0x30

	lda	##4
	bit	<aiflag
	bne	$ymaj		; br if y major.
	lda	<aimjsz
	dec	a
	sta	<temp0
	lda	<aimnsz
	dec	a
	sta	<temp1
	ldx	<aixorg
	lda	##2
	bit	<aiflag
	beq	$1
	sec
	txa
	sbc	<temp0
	tax
$1:	stx	<temp2
	ldx	<aiyorg
	lda	##1
	bit	<aiflag
	beq	$2
	sec
	txa
	sbc	<temp1
	tax
$2:	stx	<temp3
	brl	$ok

$ymaj:
	lda	<aimnsz
	sta	<temp0
	lda	<aimjsz
	sta	<temp1
	ldx	<aixorg
	lda	##1
	bit	<aiflag
	beq	$10
	sec
	txa
	sbc	<temp0
	tax
$10:	stx	<temp2
	ldx	<aiyorg
	lda	##2
	bit	<aiflag
	beq	$20
	sec
	txa
	sbc	<temp1
	tax
$20:	stx	<temp3
$ok:
	jsl	>0,Gcoor
	sec
	txa
	sbc	<temp2
	tax

	sec
	tya
	sbc	<temp3
	tay

$w:	bit	Dpdone-1
	bvc	$w

	lda	<temp0
	sta	0xfe00
	lda	<temp1
	sta	0xfe02
	stx	0xfe04
	sty	0xfe06

	lda	<temp2
	sta	Xcap
	lda	<temp3
	sta	Ycap

	sta	DaBbc

	plp
	rtl
	

	end
