/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* Alpha Diagnostics Console IO Framework */

#undef TRACE_ENABLE		/* nb be aware of complications */

#include "lib.h"
#include "uilib.h"
#include "info.h"

#include "platform.h"
#include "northbridge.h"
#include "southbridge.h"

#include "pci.h"
#include "sromport.h"
#include "uart.h"
#include "lpt.h"
#include "kbd.h"
#include "vga.h"
#include "nvram.h"
#include "cmos_rtc.h"
#include "ledcodes.h"
#include "beepcodes.h"

/*----------------------------------------------------------------------*/
/* Private data */

static io_grp curtty = GRP_NONE;
static unsigned curlog = GRP_NONE;

static io_dev con_stdin = DEV_NULL;
static io_dev con_stdout = DEV_NULL;
static unsigned con_stderr = DEV_NULL;


/*----------------------------------------------------------------------*/
/* Lowest level of abstraction in console IO - setup and nonblocking IO */

/* Private state */

typedef enum { UNKNOWN, SETUP, FAILED, NOSUCHDEV } devstate;

typedef struct { io_dev d; devstate s; } state_desc;

static state_desc init_state[] = {
	{ DEV_SROM, UNKNOWN },
	{ DEV_COM1, UNKNOWN },
	{ DEV_COM2, UNKNOWN },
	{ DEV_COM3, UNKNOWN },
	{ DEV_COM4, UNKNOWN },
	{ DEV_LPT1, UNKNOWN },
	{ DEV_LPT2, UNKNOWN },
	{ DEV_VGA,  UNKNOWN },
	{ DEV_KBD,  UNKNOWN },
	{ DEV_ROM,  UNKNOWN },
	{ DEV_NULL, SETUP }
};

static state_desc *dev2desc( io_dev D )
{
    state_desc *S;
    unsigned i;
    for ( i=0, S=init_state; i < ARRAYLEN(init_state); i++, S++ )
    {
	if ( S->d == D )	return S;
    }
    return NULL;
}

void simple_putstr( String S, io_dev d )
{
    char c;
    while( (c = *S++) != 0 )
	io_dev_putc( d, c );
}



DBM_STATUS io_dev_init( io_dev D )
{
    DBM_STATUS rval=STATUS_FAILURE;
    unsigned iters;
    unsigned baud;
    int cmos_baud;

    state_desc *Dstate = dev2desc( D );

    if ( Dstate == NULL )	return STATUS_FAILURE;
    if ( Dstate->s == SETUP )	return STATUS_SUCCESS;

    switch( D ) {
	case DEV_SROM:
	    rval = SromPortInit( );
	    break;

	case DEV_COM1: case DEV_COM2: case DEV_COM3: case DEV_COM4:
#ifdef CONFIG_TINOSA
	    baud = 7200;
#else
	    baud = 9600;				/* Default setting */
#endif

	    /* There's another variable for COM2 but don't bother with that */
	    cmos_baud = rtc_envrd( RTC_COM1_BAUD );
	    if ( cmos_baud != -1 )
		baud = cmos_baud;


	    rval = UartInitLine( D, baud );
	    break;

	case DEV_LPT1: case DEV_LPT2:
	    rval = initLpt( D );
	    break;

	case DEV_KBD:
	    /* FIXME: */
	    /* issue here - keyboard doesn't always initialise successfully
	     * with this routine as it stands.  So, I put in some retries and
	     * it is logged as a minor error if a retry is required */

#define MAX_KB_RETRIES	3			/* only used for this loop */

	    for ( iters=1; iters <= MAX_KB_RETRIES; iters++ )
	    {
	        rval = kbd_init( );

		if ( rval == STATUS_SUCCESS ) 			/* OK! */
			break;

		/* otherwise, a minor problem (or total failure), log */
		if ( iters == MAX_KB_RETRIES )
		    mobo_logf( LOG_CRIT "KEYBOARD: initialisation failed\n" );
	 	else 	/* go round again */
		    mobo_logf( LOG_WARN
				"KEYBOARD: retrying initialisation...\n" );
	    }

#undef MAX_KB_RETRIES				/* for tidiness */

	    break;

	case DEV_VGA:
	    if ( VgaDeviceDetected == TRUE )
		rval = InitVGA( );
	    else
		rval = STATUS_FAILURE;
	    break;

	case DEV_ROM:				/* non-volatile log record */
	    rval = nvlog_open( NVLOG_WRITE );
	    break;
	
	case DEV_NULL:
	    rval = STATUS_SUCCESS;
	    break;

	default:
    }
#ifdef TRACE_ENABLE
    simple_putstr( "Hello and test\n\r", D );
#endif
    if ( rval == STATUS_SUCCESS )
	Dstate->s = SETUP;
    return rval;
}


void io_dev_fini( io_dev D )
{
    state_desc *Dstate = dev2desc( D );

    if ( Dstate == NULL )			return;

    switch ( D )
    {
	/* The ROM log is opened for reading, or for writing, or not at all. */
	/* These operations are mutually exclusive. */
	case DEV_ROM:
	    nvlog_close( );
	    Dstate->s = UNKNOWN;
	    break;

	default:
	    break;
    }
}


DBM_STATUS io_dev_putc( io_dev D, int c )
{
    DBM_STATUS sval = STATUS_SUCCESS;

    /* If an error occurrs during the character IO mechanism, we could fail
     * endlessly and be unable to put out a character to describe it.
     * The solution here is to silently drop output if we are recursing and
     * bring a console offline if it fails.  We don't get any output generated
     * whilst the first IO error is being handled but we should get a message
     * back and keep going.
     */

    static uint8 recursing = FALSE;

    if ( recursing )
	return STATUS_SUCCESS;	/* Silently discard IO during other IO ops */
    recursing = TRUE;

    switch( D ) {
	case DEV_SROM:
	    SromPutChar( c );
	    break;

	case DEV_COM1: case DEV_COM2: case DEV_COM3: case DEV_COM4:
	    UartPutChar( D, c );
	    break;

	case DEV_LPT1: case DEV_LPT2:
	    putcLpt( D, c );
	    break;

	case DEV_VGA:
	    vgaputc( c );
	    break;

	case DEV_ROM:
	    sval = nvlog_putc( c );
	    break;

	case DEV_NULL:
	default:
	    break;
    }

    recursing = FALSE;

    if ( sval == STATUS_FAILURE )
    {
	io_dev_shutdown( D );
	BeepCode( beep_k_noconsole );
	outled( led_k_console_failed );
	mobo_logf( LOG_CRIT
	    "IO: an error occurred on IO dev %d and it has been stopped\n", D);
    }

    return sval;
}

int io_dev_getc( io_dev D )
{
    switch( D ) {
        case DEV_SROM:
            return SromGetChar( );

        case DEV_COM1: case DEV_COM2: case DEV_COM3: case DEV_COM4:
            return UartGetChar( D );

        case DEV_KBD:
            return kbd_getc( );

	case DEV_NULL:
        default:
	    return -1;
    }
}


#define POLL_USECS 5

int io_dev_getc_t( io_dev D, int msec)
{
    int rval=-1;
    int usec;
    unsigned long start, end, elapsed;
    int elapsed_usecs = POLL_USECS;		/* nominal start value */

    for ( usec = msec * 1000; usec >= 0; usec -= elapsed_usecs )
    {
	start = rpcc();
        rval = io_dev_getc( D );
        end = rpcc();
	if ( end <= start )
		end += 1UL << 32;		/* counter wraparound */
	elapsed = end - start;			/* elapsed cycles */
	elapsed_usecs = (elapsed * primary_impure->CYCLE_CNT) / 1000000UL;

        if ( rval != -1 )
	       break; 	          		/* a keypress */

	/* Discount the elapsed time incurred by doing the IO above */
	if ( elapsed_usecs < POLL_USECS )
	{
	    usleep( POLL_USECS - elapsed_usecs );
	    elapsed_usecs = POLL_USECS;
	}
    }
    return rval;				/* return data or timeout */
}


void io_dev_shutdown( io_dev D )
{
    if ( D == DEV_NULL || D == DEV_ERRDEV )
	return;

    if ( (con_stdout & D) == D )
	con_stdout = con_stdout & ~D;

    if ( (con_stdin & D) == D )
	con_stdin = con_stdin & ~D;

    if ( con_stderr != DEV_NULL && ((con_stderr & D) == D) )
	con_stderr = con_stderr & ~D;
}


/*----------------------------------------------------------------------*/
/* Mid-level abstraction in console IO implementation */


typedef struct { io_grp g; io_dev in, out; } group_desc;

static group_desc group[] = {
	{ GRP_SROM, DEV_SROM, DEV_SROM },
	{ GRP_COM1, DEV_COM1, DEV_COM1 },
	{ GRP_COM2, DEV_COM2, DEV_COM2 },
	{ GRP_COM3, DEV_COM3, DEV_COM3 },
	{ GRP_COM4, DEV_COM4, DEV_COM4 },
	{ GRP_LPT1, DEV_NULL, DEV_LPT1 },
	{ GRP_LPT2, DEV_NULL, DEV_LPT2 },
	{ GRP_LOCAL,DEV_KBD,  DEV_VGA  },
	{ GRP_ROM,  DEV_NULL, DEV_ROM  },
	{ GRP_NONE, DEV_NULL, DEV_NULL }
};

static group_desc *grp2desc( io_grp G )
{
    group_desc *GD;
    unsigned i;

    for ( i=0, GD=group; i < ARRAYLEN(group); i++, GD++ )
    {
	if ( GD->g == G )	return GD;
    }

    return NULL;
}


/* Return the input and output components of an IO grouping */
io_dev grp_output( const io_grp G )
{
    return grp2desc( G )->out;
}

io_dev grp_input( const io_grp G )
{
    return grp2desc( G )->in;
}


DBM_STATUS setty( io_grp G )
{
    group_desc *GD = grp2desc( G );

    if ( GD == NULL )			return STATUS_FAILURE;

    if ( io_dev_init( GD->out ) != STATUS_SUCCESS ||
	 io_dev_init( GD->in )  != STATUS_SUCCESS )
    {
	return STATUS_FAILURE;
    }

    /* setup of new tty IO-group worked, lets switch to it */

    if ( con_stdout != GD->out ) 	mobo_cls();
    curtty = G;
    con_stdin = GD->in;			/* some handy shortcuts */
    con_stdout = GD->out;

    return STATUS_SUCCESS;
}

io_grp getty( void )
{
    return curtty;
}


/* Log stream can now be multiple destinations, specified by a bitmask */
/* Care needs to be taken to observe when we're operating on IO groupings 
 * and when we're operating on single IO devices */
DBM_STATUS setlog( unsigned grp_newlogs )
{
    unsigned dev_stderrmask;
    io_grp grp_bmask; 
    group_desc *GD;
    unsigned grp_newlogs_ok = GRP_NONE;
    unsigned grp_going_down, grp_coming_up, grp_staying_up;


    dev_stderrmask = con_stderr;
    grp_coming_up = grp_newlogs & ~curlog;
    grp_staying_up = grp_newlogs & curlog;

    /* For any routings that are coming online, bring them up */
    for ( grp_bmask=1; grp_bmask <= grp_coming_up; grp_bmask <<= 1 )
    {
	/* Skip this io_dev? */
	if ( (grp_bmask & grp_coming_up) == 0 )
		continue;	

	GD = grp2desc( grp_bmask );
	if ( GD == NULL )		return STATUS_FAILURE;

	/* If device init was successful, add this channel to the new output */
	if ( io_dev_init( GD->out ) != STATUS_FAILURE )
	{
	    dev_stderrmask |= GD->out;	
	    grp_newlogs_ok |= grp_bmask;
	}
    }


    /* at this point, all selected io_grps are setup and working: TRANSFER */
    grp_going_down = curlog & ~(grp_newlogs_ok | grp_staying_up);


    /* For any routings that are going offline, shut them down */
    for ( grp_bmask=1; grp_bmask <= grp_going_down; grp_bmask <<= 1 )
    {
	/* not this io_grp? */
	if ( (grp_bmask & grp_going_down) == 0 )
		continue;

	GD = grp2desc( grp_bmask );
	io_dev_fini( GD->out );

	dev_stderrmask &= ~GD->out;		/* remove another stderr */
    }

    curlog = grp_newlogs_ok | grp_staying_up;
    con_stderr = dev_stderrmask;

    /* a bit of header info for the new log stream... */
    info_log();
    return ( curlog == grp_newlogs ) ? STATUS_SUCCESS : STATUS_WARNING;
}

io_grp getlog( void )
{
    return curlog;
}



/*----------------------------------------------------------------------*/
/* Highest-level abstraction in console IO implementation */


void PutChar( char c )
{
    if ( (con_stdout == DEV_NULL) || (con_stdout == DEV_ERRDEV) )
    		return;

    /* Special cases for PutChar */
    switch( c ) {

	case '\r': 				/* catch 'soft' CRs here */
	    mobo_goto( soft_cr );
	    return;

	case '\n':
	    cursor.x = 0, cursor.y++;
	    mobo_goto( cursor );		/* side effect: soft CR posn */
	    return;
	
	default:
	    if ( isprint( c ) )		cursor.x++;	/* avoid ANSI codes */
    }

    io_dev_putc( con_stdout, c );
}


static void dologputc( const char c )
{
    unsigned bmask;

    if ( (con_stderr == DEV_NULL) || (con_stderr == DEV_ERRDEV) )
		return;

    for ( bmask=1; bmask <= con_stderr; bmask <<= 1 )
    {
	/* use this io_dev or not? */
	if ( (con_stderr & bmask) == 0 )	continue;	
	io_dev_putc( bmask, c );
    }
}

void LogChar( char c )
{
    /* special case characters: make sure that \n, \r are made into \n\r */
    switch (c) {
    case '\n':
    case '\r':
	dologputc( '\n' );
	dologputc( '\r' );
        return;

    default:
	dologputc( c );
	return;
    }
}


/* GetChar with timeout in msec.  If zero, it should just do a single poll.
 * if no char is available, -1 is returned. 
 */

int GetChar_t( int msec )
{
    return io_dev_getc_t( con_stdin, msec );
}

int GetChar( void )
{
    int rval;

    do {
	rval = io_dev_getc_t( con_stdin, 0 );
    } while ( rval == -1 );		/* poll until keypress */

    return rval;
}

