/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* A tool for unpacking ELF format files pre-loaded into memory
 *
 * (c) 1999 Alpha Processor Inc, begun 25 January 1999 by Stig Telfer */
/*----------------------------------------------------------------------*/

#include <elf.h>

#include "lib.h"
#include "uilib.h"


/*----------------------------------------------------------------------*/
/* Data definitions */

static Elf64_Ehdr *Ehdr;
static Elf64_Phdr *Phtab;
static unsigned nphdrs;


int elf_unpack( char *src, char **dest )
{
    char *pdata;
    size_t psize, pmemsize;
    int padsz;
    uint64 addr;


    /*----------------------------------------*/
    /* validate Ehdr - quite likely input may not be ELF */

    Ehdr = (Elf64_Ehdr *)src;

    if( strncmp(ELFMAG, Ehdr->e_ident, SELFMAG) != 0)
    {
	    mobo_logf( LOG_WARN "ELF: Input file not an ELF file\n");
	    return -1;
    }

    if( (Ehdr->e_ident[EI_CLASS] != ELFCLASS64) 	||
	(Ehdr->e_ident[EI_DATA] != ELFDATA2LSB) 	||
	(Ehdr->e_type != ET_EXEC)			||
	(Ehdr->e_machine != EM_ALPHA) )
    {
	    mobo_logf( LOG_CRIT "ELF: input is not Alpha, not 64-bit, "
		    "not little-endian or not executable\n");
	    return -1;
    }


    /*----------------------------------------*/
    /* Traverse sections to the one we want */

    nphdrs = Ehdr->e_phnum;
    if (nphdrs != 1)
	    mobo_logf( LOG_WARN "ELF: Warning - %d phdrs found (want 1)\n",
		nphdrs );

    Phtab = (Elf64_Phdr *)(src + Ehdr->e_phoff);
    pdata = src + Phtab->p_offset;
    psize = Phtab->p_filesz;
    pmemsize = Phtab->p_memsz;

    /* We need to do a little dance here to strip any KSEG bits and add our
     * own choice of superpage bits to the address */

    addr = (uint64)Phtab->p_vaddr;
    addr &= ~KSEG43;			/* also strips out KSEG48... */
    addr |= DBM_SUPER;

    if ( addr != (uint64)*dest )
    {
	if ( *dest != NULL )
	    mobo_logf( LOG_WARN
		"ELF: Overriding suggested destination 0x%08lX with 0x%08lX\n",
		*dest, addr );

	*dest = (char *)addr;
    }

    mobo_logf( LOG_INFO "ELF: Writing %d bytes from image to 0x%X\n",
		 (int)psize, *dest );
    memcpy( *dest, pdata, psize );


    /* pad for BSS and round up to an 8-byte boundary */
    padsz = pmemsize & 0x7;			/* 8-byte remainder */
    if ( padsz )	padsz = 8 - padsz;	/* makes up the difference */

    if ( psize != pmemsize + padsz )		/* ie padding req'd */
    {
	mobo_logf( LOG_INFO "ELF: Adding %d bytes of BSS and %d bytes of padding\n",
		pmemsize - psize, padsz );

	padsz = pmemsize + padsz - psize;	/* grand total */
	memset( *dest + psize, 0, padsz );	/* zero out */
    }

    mobo_logf( LOG_INFO "ELF: Successful unpacking of ELF image to 0x%08X\n", 
		*dest);
    return pmemsize;
}
