/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* Alpha Diagnostics NVRAM structure interface */
/* Theory: 
 *
 * Diags uses the NVRAM for two purposes: log output and environment 
 * These two uses warrant different sectors in the NVRAM, so that any
 * update to one will not cause the other to get erased/corrupted etc.
 */

#undef TRACE_ENABLE  			/* define this for verbose debug */

#include "lib.h"
#include "uilib.h"
#include "platform.h"
#include "cmos_rtc.h"
#include "beepcodes.h"
#include "rom.h"
#include "nvram.h"


/*--------------------------------------------------------------------*/
/* NVRAM log private data and functions */


/* ROM bytes are this when they are unused */
#define NV_EOF	0xFFU

/* If less than this number of bytes free, roll over on adding a new session */
#define ROLLOVER_THRESHOLD	512	


static uint8 *slurpptr = NULL;	/* Pointer to base in RAM */
static unsigned romptr = 0;	/* Pointer to base in ROM */
static unsigned logptr = 0;	/* Offset useable for both base addresses */
static size_t logsize = 0;	/* length of log region in bytes */


#ifdef TRACE_ENABLE
static void dump_slurp( void )
{
#define DUMP_ROWS 4
#define DUMP_COLS 16

    int i, j;
    TRACE( "Slurped log record begins with:\n" );
    
    for( i=0; i<DUMP_ROWS; i++ )
    {
	mobo_logf( LOG_DBG );

	for( j=0; j<DUMP_COLS; j++ )
	    mobo_logf("%02X ", slurpptr[ i*DUMP_COLS + j ] );
	for( j=0; j<DUMP_COLS; j++ )
	{
	    mobo_logf("%c",
		isprint( slurpptr[ i*DUMP_COLS + j ] ) ? 
			slurpptr[ i*DUMP_COLS + j ]    :
			'?' );
	}

	mobo_logf( "\n" );
    }
}
#endif


/* Read the entire log sector into memory for easier manipulation */
/* Reset the log pointer to the start of the log record */
static DBM_STATUS slurp_log( void )
{

    /* Sanity Validation */
    if ( logsize == 0 )			return STATUS_FAILURE;


    /* Check whether we already have a slurped log record */
    /* (if so, scrap and recreate it) */
    if ( slurpptr != NULL )
	slurpptr = realloc( slurpptr, logsize );
    else
	slurpptr = malloc( logsize );

    if ( slurpptr == NULL )			return STATUS_FAILURE;


    /* Now copy in from ROM the entire log sector (used or empty) */
    for ( logptr=0; logptr<logsize; logptr++ )
    {
	slurpptr[ logptr ] = plat_inromb( romptr + logptr );
    }

#ifdef TRACE_ENABLE
    dump_slurp();
#endif
    logptr = 0;			/* Log pointer reset to start of record */
    return STATUS_SUCCESS;
}



/* Erase and re-slurp the log region */
/* Note: there will be problems if the log shares a sector with other things! */
static DBM_STATUS log_rollover( void )
{
    DBM_STATUS sval;

    /* Sanity Validation */
    if ( logsize == 0 )			return STATUS_FAILURE;

    sval = plat_romerase( romptr, romptr + logsize - 1, NULL, 0 );
    if ( sval == STATUS_FAILURE )		return STATUS_FAILURE;

    sval = slurp_log( );
    if ( sval == STATUS_FAILURE )		return STATUS_FAILURE;

    return STATUS_SUCCESS;
}



/* Add a magic (but human-readable) DBLX log header at the current pointer */
static DBM_STATUS add_log_record( enum log_record_opts opt )
{
    char buf[ 160 ];			/* String to contain the new msg */
    char datebuf[ 100 ];
    char *ptr;
    DBM_STATUS sval;

    datestamp( datebuf );
    sprintf_dbm( buf, NVMAGIC "%s session at '%s'\n",
	opt == NVLOGREC_NEW ? "New" : "Roll-over", datebuf );

    /* Because of ROLLOVER_THRESHOLD, we should never run out of log space
     * adding a new log record header.  If we did, it could look a bit messy. */
    for ( ptr=buf; *ptr != '\x00'; ptr++ )
    {
	sval = nvlog_putc( *ptr );
	if ( sval == STATUS_FAILURE )			return STATUS_FAILURE;
    }

    return STATUS_SUCCESS;
}


/* Record manipulation routines */

/* Next_hdr - return a pointer to the next log record, or NULL if none */
/* NOTE: has the important side-effect of setting logptr to the first byte
 * of the next record, or empty space, or the end of the log region */
static uint8 *next_hdr( uint8 *current )
{
    uint8 *rval;
    size_t magiclen = strlen( NVMAGIC );

    if ( current == NULL )
	logptr = 0;
    else
	logptr = current - slurpptr + magiclen;

    TRACE( "Looking for another header, beginning at offset 0x%X\n", logptr );

    /* Index from the earliest possible location for next record */
    for ( rval=NULL; logptr < logsize; logptr++ )
    {
        if ( slurpptr[ logptr ] == NV_EOF )
	{
	    TRACE( "Reached end of log record at 0x%X\n", logptr );
            break;
	}

        if ( strncmp( slurpptr + logptr, NVMAGIC, magiclen ) == 0 )
        {
	    TRACE( "Found new header at offset 0x%X\n", logptr );
            rval = slurpptr + logptr;
            break;
	}
    }
    return rval;
}

static size_t reclen( uint8 *hdr )
{
    size_t start;
    if ( hdr == NULL )
	hdr = slurpptr;

    start = hdr - slurpptr;
    next_hdr( hdr );		/* discard the result but use changed logptr */
    return logptr - start;
}



/*--------------------------------------------------------------------*/
/* NVRAM log public functions and data */



enum log_opts nvlog_state = NVLOG_CLOSED;


/* Read log record, setup data structures but don't write any extra details */
DBM_STATUS nvlog_open( enum log_opts flags )
{
    rom_t *H;
    DBM_STATUS sval;
    int i;

    TRACE("Setting up, flags are %d...\n", flags );

    /*----------------------------------------------------------------*/
    /* Validation: do we have a log region at all? */

    if ( (flags != NVLOG_READ) && (flags != NVLOG_WRITE) )
						return STATUS_FAILURE;

    TRACE( "Looking up our ROM header\n" );
    H = rom_search( FW_NVLOG, plat_rom_rsvd );
    TRACE( "Found at 0x%016lX\n", H );
    if ( H == NULL )				return STATUS_FAILURE;

    romptr = H->rstart;			/* Start of log in ROM */
    logsize = H->rsize;
    TRACE( "Log region is from 0x%X, %dKB\n", romptr, logsize >> 10 );

    /* Is the log region readable? */
    TRACE( "Slurping...\n" );
    sval = slurp_log( );
    if ( sval == STATUS_FAILURE )		return STATUS_FAILURE;
    TRACE( "Slurp done\n" );


    /*----------------------------------------------------------------*/
    /* If we are reading, then the log record is open for business! */
    if ( flags == NVLOG_READ )
    {
	nvlog_state = NVLOG_READ;
	return STATUS_SUCCESS;
    }


    /*----------------------------------------------------------------*/
    /* Otherwise we are writing: */

    TRACE( "We are writing...\n" );

    /* Gain the ability to write to the log during the opening process */
    nvlog_state = NVLOG_OPENING;

    /* First find the end of used space */
    /* This equates to being the last byte with value NV_EOF */
    /* For reasons of paranoia, we check the above is true */
    for ( logptr=0; logptr < logsize; logptr++ )
    {
	/* End of written ROM reached? */
	if ( slurpptr[ logptr ] == NV_EOF )
	    break;
    }
    
    for ( i=logptr; i < logsize; i++ )
    {
	if ( slurpptr[ i ] != NV_EOF )
	{
	    /* We have the ROM log sector being written, then blank, then 
	     * written some more.  This is confusing and will lead to 
	     * corruption, possibly hanging the machine */
	    BeepCode( beep_k_romlog_corrupt );
	    mobo_logf( LOG_CRIT "ROM: log corruption: log ptr was 0x%X, "
			"but non-empty log found at 0x%X\n", logptr, i );
	    sleep( 20 );			/* DEBUG DEBUG FIXME */
	    sval = log_rollover( );
	    if ( sval == STATUS_FAILURE )
	    {
		nvlog_close();
		return STATUS_FAILURE;
	    }
	    break;
	}
    }

    TRACE( "Found end of log pointer to be 0x%X\n", logptr );

    /* Wipe and regenerate the log if close to the end of available space */
    if ( logptr >= (logsize - ROLLOVER_THRESHOLD) )
    {
	TRACE( "Opting to rollover\n", logptr );
	sval = log_rollover( );
	if ( sval == STATUS_FAILURE )
	{
	    nvlog_close();
	    return STATUS_FAILURE;
	}
	TRACE( "Rollover successful, logptr = 0x%x\n", logptr );
    }


    /* Add a new log record */
    TRACE( "Adding new log record at logptr = 0x%X\n", logptr );
    sval = add_log_record( NVLOGREC_NEW );
    if ( sval == STATUS_FAILURE )
    {
	nvlog_close();
	return STATUS_FAILURE;
    }

    TRACE( "All done!\n", logptr );
 
    /* We're done, go home! */
    nvlog_state = NVLOG_WRITE;
    return STATUS_SUCCESS;
}


void nvlog_close( void )
{
    /* Free off any heap usage and other tranklements */
    if ( slurpptr != NULL )
	free( slurpptr );

    nvlog_state = NVLOG_CLOSED;
}



/* append a character to the current end of the log record, rolling over 
 * if the end of the NVRAM sector is reached */
DBM_STATUS nvlog_putc( const char c )
{
    DBM_STATUS sval;

    /* Validation: are we open for writing? (which we must be) */
    if ( (nvlog_state != NVLOG_WRITE) && (nvlog_state != NVLOG_OPENING) )
						return STATUS_FAILURE;

    /* I don't want confusing effects caused by writing magic chars */
    /* Discard these silently */
    if ( c == NV_EOF || c == '\x00' )
						return STATUS_SUCCESS;

    /* Check to see if the log buffer is full */
    if ( logptr >= logsize )
    {
	sval = log_rollover( );
	if ( sval == STATUS_FAILURE )		return STATUS_FAILURE;

	/* NOTE: this function calls us again to do the putting */
	add_log_record( NVLOGREC_ROLLOVER );
    }

    /* Put the byte out onto the ROM (and local slurp buffer too) */
    sval = plat_outromb( romptr + logptr, c );
    if ( sval == STATUS_FAILURE )		return STATUS_FAILURE;
    slurpptr[ logptr ] = c;
    logptr++;

    return STATUS_SUCCESS;
}



void nvlog_next( nvlog_t *current, nvlog_t *next )
{
    TRACE( "Running, current = 0x%X\n", current );

    /* Check if we are open for reading (which we must be) */
    if ( nvlog_state != NVLOG_READ )
    {
	TRACE( "Apparently we're not opened for reading, quitting.\n" ); 
	next->rec = NULL;
	return;
    }

    if ( (current == NULL) || (current->rec == NULL) )
    {
	TRACE( "Looking for first header in next_hdr\n" );
	next->rec = next_hdr( NULL );
	TRACE( "Got 0x%X\n", next->rec );
    }
    else
    {
	/* Otherwise, find the next magic header entry, of one exists */
	next->rec = next_hdr( current->rec );
	TRACE( "Looking for next header, got 0x%X\n", next->rec );
    }
    next->len = reclen( next->rec );
    TRACE( "Got a length of %d\n", next->len );
    return;
}

