/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* Floppy disk write/read tests */
/* Currently there is only a destructive floppy test, ie after this your
 * hapless disk will be totally trashed.  Be warned!
 */


#include "lib.h"		/* Digital Debug Monitor */
#include "uilib.h"
#include "northbridge.h"
#include "platform.h"

#include "nttypes.h"
#include "floppy.h"



/*----------------------------------------------------------------------*/
/* Private data definitions */

/* disk geometry and interface organisation, 1 blk = 1 char on screen */
#define NSECTORS	2880	/* number of sectors on disk */
#define NBLKS		288	/* number of characters on screen */
#define SECTORS_PER_BLK	10
#define BLKS_PER_ROW	72
#define BLK_ROWS	4
#define TEST_BLKS	10	/* number of blks to test at a time */
#define TEST_PASSES	10

#define SECTOR_SZ	512	/* bytes per sector on disk */
#define BLK_SZ		(SECTOR_SZ * SECTORS_PER_BLK)
#define VECTOR_SZ	(BLK_SZ * TEST_BLKS)

static FLOPPY_DRIVE_INFO d = {
    HD_DRIVE_TYPE,
    HD_DRIVE_TYPE,
    0,			/* Current track */
    0,			/* =1 for non-DMA (NOTE: doesn't actually work)*/
    0			/* drive number */
};


typedef unsigned char bstate_t;
typedef unsigned char mstate_t;



/*----------------------------------------------------------------------*/
/* User interface control.  By writing to this function, the state of the 
 * floppy can be described consistently and output mechanism tuned. 
 */

static Point p_state, m_state;
static unsigned char passno;

static DBM_STATUS init_state( void );
static DBM_STATUS fini_state( void ); 

static void floppy_fail( const String msg );
static void set_blk_state( const unsigned blkno, const bstate_t newstate );
static void set_msg_state( const mstate_t newstate );

static void floppy_fail( const String msg )
{
    mobo_logf( LOG_CRIT "FLOPPY: drive failure: %s\n", msg );
    mobo_alertf( "Floppy drive failure", "Error reported: %s", msg );

    fini_state();		/* wind things down */
}


static const String blkstate[] = {

#define BLK_UNTESTED	0
	FG_DGREY BLK_HALF CU_NORMAL,

#define BLK_WRITTEN	1
	FG_YELLOW BLK_FULL CU_NORMAL,

#define BLK_READ	2
	FG_GREEN BLK_FULL CU_NORMAL,

#define BLK_FAILED	3
	FG_RED BLK_FULL CU_NORMAL
};

static bstate_t *state_array;


static void set_blk_state( const unsigned blkno, const bstate_t newstate )
{
    Point P;

    state_array[ blkno ] = newstate;
    
    P.x = p_state.x + blkno % BLKS_PER_ROW;
    P.y = p_state.y + blkno / BLKS_PER_ROW;

    mobo_goto( P );
    printf_dbm( blkstate[ newstate ] );
}




static const String msgstate[] = {

#define MSG_SETUP	0
	"Initialising drive",

#define MSG_WRITING	1
	"Writing",

#define MSG_READING	2
	"Reading",

#define MSG_SHUTDOWN	3
	"Shutting down",

#define MSG_WFAIL	4
	"Write floppy_failure!",

#define MSG_RFAIL	5
	"Read floppy_failure!",
};

#define MSG_STATE_LEN	20


static void set_msg_state( const mstate_t newstate )
{
    mobo_goto( m_state );
    printf_dbm( "Floppy status: %-20sPass %d of %d",
	msgstate[ newstate ], passno, TEST_PASSES );
}




static DBM_STATUS init_state( void )
{
    unsigned i;

    /* initialise data structures, draw the interface */
    passno = 0;
    state_array = malloc( NBLKS * sizeof( bstate_t ) );
    if ( state_array == NULL )		return STATUS_FAILURE;

    p_state.x = r_app.x + 2, p_state.y = r_app.y + 1;
    m_state.x = p_state.x, m_state.y = p_state.y + BLK_ROWS + 1;

    mobo_box( r_app, "Floppy disk read/write test");
    for ( i=0; i<NBLKS; i++ )	set_blk_state( i, BLK_UNTESTED );
    set_msg_state( MSG_SETUP );

    /* Initialise the Southbridge SuperIO controller */
    if ( InitializeFloppyDrive(&d) != STATUS_SUCCESS )
    {
	floppy_fail( "Floppy initialization floppy_failed" );
	return STATUS_FAILURE;
    }

    return STATUS_SUCCESS;
}


/* clean up the state of the floppy - don't want to leave the drive spinning */
static DBM_STATUS fini_state( void )
{
    if ( state_array != NULL )		free( state_array );
    SpinDownFloppyDrive(&d);
    return STATUS_SUCCESS;
}


/*----------------------------------------------------------------------*/
/* Disk access interface */

static void generate_pseudorandom_vector( unsigned *vector )
{
    unsigned i;

    for ( i=0; i < VECTOR_SZ / sizeof(unsigned); i++ )
	vector[ i ] = random();
}




static DBM_STATUS write_test_data( const unsigned blkbase,
				   const unsigned *wdata )
{
    DBM_STATUS sval;
    unsigned i,j;
    unsigned secbase = blkbase * SECTORS_PER_BLK;
    unsigned sec_cnt;

    set_msg_state( MSG_WRITING );

    /* Write to the disk, sector by sector, with pseudorandom data */
    for (i = 0, sec_cnt=0; i < TEST_BLKS; i++)
    {
	set_blk_state( blkbase + i, BLK_WRITTEN );

	for ( j=0; j < SECTORS_PER_BLK; j++, sec_cnt++ )
	{
	    sval = WriteLogicalSector( &d, HD_DISK_TYPE, secbase + sec_cnt,
		    (unsigned char *)wdata + sec_cnt * SECTOR_SZ);

	    if (sval != STATUS_SUCCESS)
	    {
		mobo_logf( LOG_CRIT "Error on writing to sector %d "
				    "(sector %d of this test pass)\n",
				    secbase + sec_cnt, sec_cnt );
		set_msg_state( MSG_WFAIL );
		set_blk_state( blkbase + i, BLK_FAILED );
		return STATUS_FAILURE;
	    }
	}
    }
    return STATUS_SUCCESS;
}

static DBM_STATUS read_test_data( const unsigned blkbase,
				   const unsigned *rdata )
{
    DBM_STATUS sval;
    unsigned i,j;
    unsigned secbase = blkbase * SECTORS_PER_BLK;
    unsigned sec_cnt;

    set_msg_state( MSG_READING );

    /* Write to the disk, sector by sector, with pseudorandom data */
    for (i = 0, sec_cnt=0; i < TEST_BLKS; i++)
    {
	set_blk_state( blkbase + i, BLK_READ );

	for ( j=0; j < SECTORS_PER_BLK; j++, sec_cnt++ )
	{
	    sval = ReadLogicalSector( &d, HD_DISK_TYPE, secbase + sec_cnt,
		    (unsigned char *)rdata + sec_cnt * SECTOR_SZ);

	    if (sval != STATUS_SUCCESS)
	    {
		mobo_logf( LOG_CRIT "Error on reading from sector %d "
				    "(sector %d of this test pass)\n",
				    secbase + sec_cnt, sec_cnt );
		set_msg_state( MSG_RFAIL );
		set_blk_state( blkbase + i, BLK_FAILED );
		return STATUS_FAILURE;
	    }
	}
    }
    return STATUS_SUCCESS;
}


/*----------------------------------------------------------------------*/
/* Main test routine */

DBM_STATUS floppy(void)
{
    /* I use two arrays here because it prevents any cache coherency 
     * problems going undetected. */

    unsigned wdata[VECTOR_SZ / sizeof(unsigned)];
    unsigned rdata[VECTOR_SZ / sizeof(unsigned)];

    unsigned baseblk;
    int i;
    int floppy_fails;
    DBM_STATUS sval;
    char key;


    sval = init_state( );
    if ( sval != STATUS_SUCCESS )	return sval;

    key = mobo_alertf( "Destructive floppy test",
	"This test will erase any data on your floppy disk\r"
	"Insert a blank floppy and press any key to continue, or q to quit" );

    if ( tolower( key ) == 'q' ) {
	fini_state( );
	return STATUS_SUCCESS;
    }

    /* Test procedure: pick at random a fixed number of contiguous blocks 
     * (each a fixed number of sectors), write them and then read back to 
     * verify, updating the interface all the time.
     */

    for ( passno=1; passno <= TEST_PASSES; passno++) {
	/* choose a random sequence for this test pass */
	generate_pseudorandom_vector( wdata );
	baseblk = (unsigned)random() % (NBLKS - TEST_BLKS);


	/* Write out */

	sval = write_test_data( baseblk, wdata );
	if ( sval != STATUS_SUCCESS )
	{
	    floppy_fail( "Writing test data" );
	    return STATUS_FAILURE;
	}


	/* Read back */

	sval = read_test_data( baseblk, rdata );
	if ( sval != STATUS_SUCCESS )
	{
	    floppy_fail( "Reading test data" );
	    return STATUS_FAILURE;
	}


	/* Compare - count all errors in a pass before reporting */

	for ( i=0, floppy_fails=0; i < VECTOR_SZ / sizeof( unsigned ); i++ )
	    if ( wdata[ i ] != rdata[ i ] )	/* Error case */
		floppy_fails++;
	
	if ( floppy_fails == 0 )			continue;

	/* List errors */
	for ( i=0; i < VECTOR_SZ / sizeof( unsigned ); i++ )
	{
	    if( wdata[ i ] != rdata[ i ] ) 
	    {
		unsigned sector = 
			(baseblk*BLK_SZ + i*sizeof(unsigned)) / SECTOR_SZ;
		unsigned offset = (i*sizeof( unsigned )) % SECTOR_SZ;

		mobo_logf( LOG_CRIT "FLOPPY: Sector %d, offset %d: "
		    "wrote 0x%08X, read 0x%08X, diff 0x%08X\n",
		     sector, offset, wdata[i], rdata[i], wdata[i] ^ rdata[i] );
 
		key = mobo_alertf( "Floppy disk data corruption",
		    "Sector %d, offset %d: "
		    "wrote 0x%08X, read 0x%08X, diff 0x%08X\r"
		    "%d errors detected this pass.  "
		    "Any key to continue or q to quit now",
		    sector, offset, wdata[i], rdata[i],
		    wdata[i] ^ rdata[i], floppy_fails );
		    
		if ( tolower( key ) == 'q' )
		{
		    fini_state();
		    return STATUS_FAILURE;
		}
	    }
	}
    }

    fini_state();
    mobo_alertf("Tests Passed", "No errors were found with the floppy");
    return STATUS_SUCCESS;
}
