#ifndef	DIAGS__NVRAM_H_LOADED
#define	DIAGS__NVRAM_H_LOADED
/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* Alpha Diagnostics NVRAM structure interface */
/* Theory: 
 *
 * Diags uses the NVRAM for two purposes: log output and environment 
 * These two uses warrant different sectors in the NVRAM, so that any
 * update to one will not cause the other to get erased/corrupted etc.
 */


enum log_opts { NVLOG_CLOSED=0, NVLOG_READ=1, NVLOG_WRITE=2, NVLOG_OPENING=3 };

enum log_record_opts { NVLOGREC_NEW, NVLOGREC_ROLLOVER };


/* This variable is used to record whether the log record is in use at all */
extern enum log_opts nvlog_state;


/* log stream IO routines */

extern DBM_STATUS nvlog_open( enum log_opts );
extern void nvlog_close( void );
extern DBM_STATUS nvlog_putc( const char c );		/* appends to log */


/* Traversing the log record in ROM is an iterative process, achieved thus: */

typedef struct {
	uint8 *rec;
	size_t len;
} nvlog_t;

/* Passing in NULL as current will return the first log record.
 * receiving NULL as next->rec means there are no more log records to read */

extern void nvlog_next( nvlog_t *current, nvlog_t *next );


/*--------------------------------------------------------------------*/
/* NVRAM Environment format: */
/* The NVRAM Environment consists of key/value pairs, separated by an '='
 * character.  
 * The end of the environment is marked by 0xFF character.
 * The end of strings is marked either by 0x00 or 0xFF.
 * This leads to an optimisation where a key/value pair can be updated without
 * erasing the NVRAM by zeroing out the old value and appending a new pair
 * at the end of the NVRAM block.
 */ 


#define NVENV_MAXKEY	256
#define NVENV_MAXVAL	256

typedef struct _NVRAM_ENTRY {
    unsigned short length;
    char *key;
    char *val;
    struct _NVRAM_ENTRY *next;
} nvenv_t;


/* Initialisation */
DBM_STATUS nvenv_init( void );

/* Update environment with new key-value pair */
DBM_STATUS nvenv_write( const String key, const String val);	

/* Remove a key-value pair from the NVRAM environment */
DBM_STATUS nvenv_remove( const String key );

/* Lookup a key, return the value */
String nvenv_lookup( const String key );

/* Care should be taken when using this function.  The NVRAM contents must
 * not be modified between starting an nvenv_nextentry( NULL ), and when the 
 * nvenv_nextentry iterative read has been finished.
 */
nvenv_t *nvenv_nextentry( nvenv_t * );	/* get next kvp, (or first if null) */


/*--------------------------------------------------------------------*/
/* Defined NVRAM Environment keys */

/* This variable is free-form.  If set, it will be interpreted as a single
 * diags command to be run on startup (if the conditions are right).
 * Popular choices might be:
 * - bios		run whatever BIOS is present in the ROM
 * - bios 2		run SRM specifically (SRM is FW ID 2, see include/rom.h)
 * - linux		boot an embedded kernel
 * - flash		run the flash firmware recovery utility
 * - diags		magic dummy command that results in diags main menu
 */

/* On entry, and in the absence of more critical events, diags does this */
#define KEY_DIAGS_ACTION "diags_action"
#define	   DA_DIAGS 	 "diags"		/* magic command to run diags */


#define KEY_AUTO_ACTION	"auto_action"		/* on power up SRM does this */
#define    AA_BOOT	"BOOT"
#define    AA_HALT	"HALT"
#define    AA_RESTART	"RESTART"

#define	KEY_BOOT_DEV		"boot_dev"
#define KEY_BOOTDEF_DEV		"bootdef_dev"
#define     BD_DBLX		"rom"
#define	    BD_COM1		"ttyS0"
#define KEY_BOOTED_DEV		"booted_dev"
#define KEY_BOOT_FILE		"boot_file"
#define	KEY_BOOTED_FILE		"booted_file"
#define	KEY_BOOT_OSFLAGS 	"boot_osflags"
#define KEY_BOOTED_OSFLAGS	"booted_osflags"
#define KEY_BOOT_RESET		"boot_reset"
#define		BR_OFF		"OFF"
#define		BR_ON		"ON"
#define KEY_DUMP_DEV		"dump_dev"
#define KEY_ENABLE_AUDIT	"enable_audit"
#define		EA_OFF		"OFF"
#define		EA_ON		"ON"
#define	KEY_LICENSE		"license"
#define	KEY_CHAR_SET		"char_set"
#define KEY_LANGUAGE		"language"
#define	KEY_TTY_DEV		"tty_dev"
#define	KEY_LOG_DEV		"log_dev"



#define KEY_TTY		"default_tty"		/* preferred consoles */
#define		TT_LOC	"local"
#define		TT_COM1	"com1"
#define		TT_COM2 "com2"
#define		TT_COM3	"com3"
#define		TT_COM4	"com4"
#define		TT_SROM	"srom"
#define		TT_NONE	"none"

#define KEY_THERM_OS	"shutdown_temp"		/* Pending proper SRM name */
#define KEY_THERM_HYST	"therm_hyst"		/* Pending proper SRM name */

/* On booting an embedded kernel, should the initrd be loaded along with it? */
#define KEY_DBLX_INITRD	"dblx_use_initrd"
#define		DB_OFF	"OFF"
#define		DB_ON	"ON"

/* Is wake-on-lan functionality enabled or disabled? */
/* This variable is stored only as the relevant bit in the NIC's EEPROM */
#define KEY_WOL_ENABLE	"wol_enable"
#define		WOL_OFF	"OFF"
#define		WOL_ON	"ON"

/* On booting an embedded kernel, should some memory be put by for bootmem
 * transfers? */
#define	KEY_ENABLE_BOOTMEM	"enable_bootmem"
#define		EB_OFF		"OFF"
#define		EB_ON		"ON"

#define KEY_BOOTMEM_ALLOC	"bootmem_alloc_mb"


/* Variables that are stored in CMOS instead of in the NVRAM */
#define KEY_COM1_BAUD	"com1_baud"		/* SRM console speed settings */
#define KEY_COM2_BAUD	"com2_baud"


/*--------------------------------------------------------------------*/
/* NVRAM/NVENV session header data structure */
/* Note: the ROM is not aligned in any particular way, this header can
 * occur on any byte boundary.
 */

#define NVMAGIC "API-DBLX-MAGIC: "



// #define NVMAGIC	0xF1F2F3F4U
#define NV_CURRENT_VERSION	1

enum NV_flags {

	/* Header v0 flags */
	/* NVRAM log buffer has overflowed and been restarted again */
	NV_ROLLOVER = (1<<0)
};

typedef union {
    struct {
	struct {				/* 16 bytes */
	    unsigned nvmagic;			/* NVRAM magic number */
	    unsigned char version;		/* implementation version */
	    unsigned char size;			/* header size in bytes */
	    unsigned char daywk, month, daymth, hour, min, sec;
	    unsigned short year;		/* Date session started */
	    unsigned short flags;		/* Attributes defined above */
	} v0;

	/* Add any new version extensions here... */

	unsigned csum;		/* header checksum, excluding this val */
    } hdr;

    unsigned char raw[1];
} nvhdr_t;


/* This data structure is used by the NVRAM log to produce a list of 
 * NVRAM headers found while scanning the log record */

typedef struct __logrec_t {
    nvhdr_t *H;
    unsigned logptr;			/* offset into log region */
    struct __logrec_t *next;
} logrec_t;

extern logrec_t *nvlog_rec;


/*--------------------------------------------------------------------*/
#endif 		/* DIAGS__NVRAM_H_LOADED */

