/*
 * misc.c
 * 
 * This is a collection of several routines from gzip-1.0.3 
 * adapted for Linux.
 *
 * puts by Nick Holloway 1993
 *
 * Adapted to Linux/Alpha boot by David Mosberger (davidm@cs.arizona.edu).
 */

#include "lib.h"
#include "uilib.h"
#include "gzip.h"
#include "init.h"

extern struct bootfs *bfs;
extern char *dest_addr;
extern long bytes_to_copy;

unsigned char *inbuf;
unsigned char *window;
unsigned outcnt;
unsigned insize;
unsigned inptr;
unsigned long bytes_out;
int method;

static char *dest_addr;
extern void *malloc(size_t);
extern void free(void * pointer);


/*--------------------------------------------------------------------*/
/* Generic C standard library widgetry */

#define SCRATCHBUF	(8UL<<20)		/* 8MB start of heap */

void *malloc( size_t nob )
{
    static char *heap = (char *)SCRATCHBUF;	/* use scratch area for heap */
    static char *oldheap;

    /* round to nearest 8-byte boundary */
    if ( nob % 8 != 0 )		nob = (nob | 0x7) + 1;

    oldheap = heap;
    heap += nob;
    return oldheap;
}

int memcmp(const void *s1, const void *s2, size_t n)
{
    const char *c1, *c2;
    for ( c1=s1, c2=s2; n>0; c1++, c2++, n-- )
	if ( *c1 != *c2 )
	    return -1;			/* sloppy - non-match merely non-zero */
    return 0;
}

void *memset(void *s, int c, size_t n)
{
    char *cs;
    for ( cs=s; n>0; cs++, n-- )		*cs = c;
    return s;
}

void *memcpy(void *dest, const void *src, size_t n)
{
    char *cdst;
    const char *csrc;

    for ( cdst=dest, csrc=src; n>0; cdst++, csrc++, n-- )	*cdst = *csrc;
    return dest;
}





void
error(char *x)
{
	init_putstr( LOG_CRIT "Unzip error -- halting.\n" );
}

/*
 * Clear input and output buffers
 */
static int block_number = 0;
void
clear_bufs(void)
{
        block_number = 0 ;
	outcnt = 0;
	insize = inptr = 0;
	bytes_out = 0;
}


/*
 * Check the magic number of the input file and update ofname if an
 * original name was given and to_stdout is not set.
 * Return the compression method, -1 for error, -2 for warning.
 * Set inptr to the offset of the next byte to be processed.
 * This function may be called repeatedly for an input file consisting
 * of several contiguous gzip'ed members.
 * IN assertions: there is at least one remaining compressed member.
 *   If the member is a zip file, it must be the only one.
 */
static int get_method(void)
{
	unsigned char flags;
	char magic[2]; /* magic header */

	magic[0] = get_byte();
	magic[1] = get_byte();

	method = -1;                 /* unknown yet */
	if (memcmp(magic, GZIP_MAGIC, 2) == 0
	    || memcmp(magic, OLD_GZIP_MAGIC, 2) == 0) {

		method = get_byte();
		flags  = get_byte();
		if ( (flags & ENCRYPTED) != 0	 || 
		     (flags & CONTINUATION) != 0 ||
		     (flags & RESERVED) != 0 )
		{
		  init_putstr( LOG_CRIT "UNZIP: Unexpected input format!\n");
		  return -1 ;
		}
		(void) get_byte();	/* skip over timestamp */
		(void) get_byte();
		(void) get_byte();
		(void) get_byte();

		(void) get_byte();	/* skip extra flags */
		(void) get_byte();	/* skip OS type */

		if ((flags & EXTRA_FIELD) != 0) {
			unsigned len = get_byte();
			len |= get_byte() << 8;
			while (len--) (void)get_byte();
		}

		/* Get original file name if it was truncated */
		if ((flags & ORIG_NAME) != 0) {
			/* skip name */
			while (get_byte() != 0) /* null */ ;
		}

		/* Discard file comment if any */
		if ((flags & COMMENT) != 0) {
			while (get_byte() != 0) /* null */ ;
		}
	}
	return method;
}



/*
 * Write the output window window[0..outcnt-1] holding uncompressed
 * data and update crc.
 */
void
flush_window(void)
{

	if (!outcnt) {
		return;
	}

	bytes_out += outcnt;

	if (dest_addr) {
		/*
		 * Simply copy adjacent to previous block:
		 */
		memcpy((char*)dest_addr, window, outcnt);
		dest_addr += outcnt;
	} 
}



int init_unzip(void *from, void *where)
{
 	int newsize;
	inbuf = (char*)from;
	window = malloc(WSIZE);
	dest_addr = (char *)where;

	clear_bufs();

	method = get_method();
	if (method < 0) {
	  init_putstr( LOG_INFO "UNZIP: this is not a zipped file\n") ;
	  return -1 ;
	}

	init_putstr( LOG_INFO "UNZIP: this is a zipped file.") ;

	newsize = unzip(0, 0);
	return newsize;
}
