/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* System-related configuration details for the API Swordfish platform */
/* Begun by Stig Telfer, Alpha Processor Inc, 29 June 1999 */

#undef TRACE_ENABLE

#include "lib.h"
#include "uilib.h"
#include "info.h"

#include "specifics.h"			/* platform-specific definitions */
#include "platform.h"			/* interface for this module */
#include "northbridge.h"		/* Generic NB access */
#include "southbridge.h"		/* Generic SB access */

#include "pci.h"

#include "osf.h"
#include "mcheck.h"
#include "cpu/ev6.h"
#include "cmos_rtc.h"
#include "beepcodes.h"
#include "hwrpb.h"			/* for systype settings */

#include "i2c.h"
#include "i2c/lm75.h"
#include "i2c/adm9240.h"

const unsigned plat_sys_type = ST_DEC_TSUNAMI;
const unsigned plat_sys_variation = 1 << 10;

static void up2k_halt_button_action( void );
static DBM_STATUS up2k_jumper_sanity( void );

const char *Prompt = "Swordfish> ";

static const PCISlot_t pcihose0[] = { 
	{ PCI0_DEC_TSU, "Tsunami" },
	{ PCI0_CYP_ISA, "Cypress ISA" },
	{ PCI0_AIC_SCSI,"AIC SCSI" },
        { PCI0_SLOT1,	"Slot 1" },
        { PCI0_SLOT2,	"Slot 2" },
        { PCI0_SLOT3,	"Slot 3" },
        { 0, 0 } };

static const PCISlot_t pcihose1[] = { 
        { PCI1_SLOT4,	"Slot 4" },
        { PCI1_SLOT5,	"Slot 5" },
        { PCI1_SLOT6,	"Slot 6" },
        { 0, 0 } };


/* note - ordering here is critical, it must match the bus numbering as 
 * decided by the PCI device tree building algorithm */

const PCIStructure_t PCIStructure[] = {
    { VARIANT_PCI, pcihose0 },			/* PCI hose 0 layout */
    { VARIANT_PCI, pcihose1 },			/* PCI hose 1 layout */
};

const short plat_pci_hose_count = ARRAYLEN( PCIStructure );


/* The I2C bus requires large chunks of the system to be working.
 * Use this variable if a function would like to read I2C data but would
 * prefer not to do large-scale machine initialisation in order to do so.
 */
static BOOLEAN i2c_online = FALSE;

/* Are we running on a Swordfish or a Swordfish+? */
/* This is an I2C bus based decision - we don't know until I2C is initialised */
static BOOLEAN is_swplus = FALSE;




DBM_STATUS plat_setup( void )
{
    /* Register early information about this platform */
    info_submit( SRC_DIAGS, SUBJ_MOTHERBOARD, "Motherboard",
			"API UP2000/UP2000+" );

    return STATUS_SUCCESS;
}


static void jblk_analyse( const unsigned cpuid, unsigned gpen )
{
    unsigned clk, bcache;
    BOOLEAN present;
    String clkstr, bcstr;

    present = gpen & 0x80 ? FALSE : TRUE;		/* bit 7=0 -> present */

    clk = gpen & 0x7U;
    switch( clk ) {
	case 5: clkstr = "667 MHz"; break;
	case 6: clkstr = "750 MHz"; break;
	default: clkstr = "Unknown MHz"; break;
    }

    bcache = (gpen >> 3) & 0x7U;

    switch( bcache ) {
	case 0x0: bcstr = "Bcache disabled"; break;
	case 0x2: bcstr = "2 MBytes"; break;
	case 0x3: bcstr = "4 MBytes"; break;
	case 0x4: bcstr = "8 MBytes"; break;
	default: bcstr = "Undocumented Bcache setting(!)"; break;
    }

    info_submit( SRC_JUMPERS, SUBJ_CPU, "Processor clock frequency",
	"CPU %d: %s", cpuid, clkstr );

    info_submit( SRC_JUMPERS, SUBJ_CPU, "L2 cache size", "CPU %d: %s",
	cpuid, bcstr );

    info_submit( SRC_JUMPERS, SUBJ_CPU, "Processor presence", "CPU %d: %s",
	cpuid, present ? "detected present" : "not detected" );
}


DBM_STATUS plat_fixup( void )
{
    unsigned long misc;
    uint8 gpen4;
    unsigned tsu_clk;
    String clk;

    /* clear any outstanding NXM interrupts incurred during startup */
    /* This is achieved by writing 1 to Cchip CSR MISC:NXM if set */

    misc = cchip[ MISC ].r;
    if ( misc & misc_m_nxm )
    {
	cchip[ MISC ].r = misc_m_nxm;		/* write that bit for W1C */
	mb();					/* make sure it goes */
    }

    /* Enable relevant interrupts in the Tsunami interrupt mask register */
    /* These are the interrupts that SRM console enables for Linux */
    cchip[ DIM0 ].r = PCI0_INT | PCI0_BUS_ERROR | PCI1_BUS_ERROR | IRQ_ERR_NXM;
    cchip[ DIM1 ].r = 0UL;


    /* Register information about this platform */
    gpen4 = tigrb( TIG_GPEN4 );
    TRACE( "TIG jumper gpen4=0x%X\n", gpen4 );
    tsu_clk = (gpen4 >> 4) & 0x7U;		/* see DP264 ref fig 5-3 */
    switch( tsu_clk )
    {
	case 0x4:
	    clk = "Clock at 83MHz";
	    break;

	case 0x1:
	    clk = "Clock at 75MHz";
	    break;

	default:
	    TRACE("tsu_clk found to be 0x%X\n", tsu_clk );
	    clk = "Not Recognised (!!)";
	    break;
    }
    info_submit( SRC_JUMPERS, SUBJ_CHIPSET,
		    "Tsunami/EV6 bus frequency", clk );

    /* Processor jumper block analysis */
    jblk_analyse( 0, tigrb( TIG_GPEN5 ) );
    jblk_analyse( 1, tigrb( TIG_GPEN6 ) );


    /* TIG PAL Revision */
    info_submit( SRC_CHIPSET, SUBJ_CHIPSET, "TIG PAL revision",
		"Rev %02X", tigrb( TIG_PALREV ) );

    /* On-board SCSI revision */
    info_submit( SRC_CHIPSET, SUBJ_CHIPSET, "Adaptec SCSI revision",
		"Rev %02X",
		pcicfgrb( 0, PCI0_AIC_SCSI, 0, PCI_CFG_REG_REVISION_ID ) );

    return STATUS_SUCCESS;
}


DBM_STATUS plat_post( void ) 
{
    DBM_STATUS sval;
    DBM_STATUS final_return = STATUS_SUCCESS;

    if ( smp_primary_cpu != 0 )
    {
        BeepCode( beep_k_hw_mismatch );

	bring_up_console();

        mobo_logf(                                                                          LOG_WARN "POST: CPU 0 is either disabled or malfunctioning!\n"
            LOG_WARN "POST: Boostrap processor is CPU %d\n", smp_primary_cpu );

	mobo_box( r_lrgapp, "Anomalous Processor Configuration" );
	printf_dbm(
	    "Diags has detected that the primary processor in this machine\r"
	    "is not CPU 0.  The system will still be capable of functioning\r"
	    "without CPU 0, but operating systems may not expect this and\r"
	    "unpredictable results may occur.\r"
	    "\r"
	    "Diags suggests that CPU slot 0 be used if possible.\r"
	    "Press any key if you wish to continue anyway." );
 
	mobo_key( 0 );
    }

    TRACE( "Checking for halt button...\n" );

    /* Check for UP2K/CH320 dodgy toggling halt button */
    /* note: this comes after the machine check test because the halt button
     * assertion raises a 'harmless' machine check variant
     */
    swpipl( 6 );        /* IPL 6 includes halt button (but not much else) */
    msleep( 1 );        /* if halt button then the handler will prompt user */
    swpipl( 7 );        /* go back to having interrupts enabled */

    TRACE( "Checking for jumper oddities...\n" );
    sval = up2k_jumper_sanity();
    if ( sval != STATUS_SUCCESS )	final_return = STATUS_FAILURE;

    TRACE( "Analysing RTC...\n" );

    /* Check for Cypress CY693 dodgy wandering timer interrupt */
    sval = up2k_rtc_analyse();
    if ( sval != STATUS_SUCCESS ) 	final_return = STATUS_FAILURE;

    TRACE( "Analysing System Health...\n" );

    /* Have a look at system health monitors for a bad environment */
    nb_setup();				/* need to have IO buses mapped? */
    sb_setup();
    plat_setup();

    nb_fixup();
    sb_fixup();
    plat_fixup();
    sval = i2c_post();
    if ( sval != STATUS_SUCCESS )
    {
        /* Bring up a console */
        BeepCode( beep_k_system_health );
        bring_up_console( );

        /* Get the details from the I2C module */
        i2c_post_report();
	final_return = STATUS_FAILURE;
    }


    /* Check to see if we're a UP2000 (determined after I2C init) running 
     * with EV68 processors (a dodgy combination)
     */
    if ( !is_swplus &&
		ev6_cpurevno( primary_impure->I_CTL ) >= EV6__EV68_REVBASE )
    {
        /* Bring up a console */
        BeepCode( beep_k_system_health );
        bring_up_console( );

	mobo_logf( LOG_CRIT "CPU: Revision later than EV67 in UP2000\n" );
	mobo_box( r_lrgapp, "CPU Revision Alert" );
	mobo_goto( p_app );
	printf_dbm(
		"This system has been detected as a UP2000 motherboard\r"
		"The primary Slot-B module was found to be:\r"
		"%s\r"
		"UP2000 motherboards cannot support processors of revision\r"
		"later than EV67.  Please power down the system or long-term\r"
		"damage may occur.", ev6_cpurev( primary_impure->I_CTL ) );

	final_return = STATUS_FAILURE;
    }

    TRACE( "Done\n" );
    return final_return;
}


DBM_STATUS plat_reset( int argc, char *argv[] )
{
    if ( is_swplus )
    {
	/* Swordfish+ has reset via I2C */
	/* To reset a Swordfish+, set the system reset bit from 1-0-1 */
	const I2Cbus_t *I;
	DBM_STATUS sval;
	uint8 csr0;

	I = i2c_lookup( I2C_REG0 );
	sval = i2cdev_read( I->addr, -1, 1, &csr0 );
	if ( sval == STATUS_FAILURE )
	{
	    mobo_logf( LOG_CRIT "RESET: Couldn't access I2C bus\n" );
	} else {
	    csr0 = csr0 | SWPL_R0_RESET;			/* set */
	    i2cdev_write( I->addr, -1, 1, &csr0 );
	    csr0 = csr0 ^ SWPL_R0_RESET;			/* clear */
	    i2cdev_write( I->addr, -1, 1, &csr0 );
	    csr0 = csr0 ^ SWPL_R0_RESET;			/* set */
	    i2cdev_write( I->addr, -1, 1, &csr0 );
	}

    } else {

	/* to reset a DP264/Swordfish family motherboard, a write to the TIGbus
	 * is required: address 801 38xx x100 (TIG_RESET) write 0, then 1 */

	/* NOTE: this code does not currently work on my UP2000... */

	tigwb( TIG_SWRST, 0 );
	msleep(100);
	tigwb( TIG_SWRST, 1 );
	msleep(100);
	tigwb( TIG_SWRST, 0 );
	mobo_logf( LOG_WARN "RESET: nothing happened via TIG!\n");

	/* It seems the above never works? */

    }

    /* Give it a little time to take effect */
    sleep( 1 );
    mobo_logf( LOG_WARN "RESET: software triggered reset did nothing!\n" );
    
    asm( "halt" );
    mobo_logf( LOG_WARN "RESET: software halt did nothing!\n" );

    return STATUS_FAILURE;			/* didn't reset OK */
}


/*----------------------------------------------------------------------*/
/* System status string, for printing out during memory stress test */
/* we have four lines of room here */

void plat_sysstat( String S )
{
    char p0stat[160], p1stat[160];
    unsigned ipl = swpipl( 7 );
    swpipl( ipl );                      /* find our IPL */

    /* Note: in error cases pchip_err_analyse_clear will print two lines */
    /* In which case we need to exchange \n for \r */
    pchip_err_analyse_clear( p0stat, 0 );
    pchip_err_analyse_clear( p1stat, 1 );
    ntor( p0stat );
    ntor( p1stat );

    sprintf_dbm( S,
	"PCI hose 0: %s\r"
	"PCI hose 1: %s\r"
	"System interrupts are at level %d (0=all enabled, 7=all disabled)",
		p0stat, p1stat, ipl );
}


/*----------------------------------------------------------------------*/
/* setup of all system interrupts - o = 0 (off), 1 (on) */

void plat_intsetup( int o ) 
{
    plat_intclr();                      /* wipe the slate clean beforehand */

    if ( o )                            /* enabling case */
    {
        mobo_logf( LOG_INFO "Enabling ISA interrupts...\n");
        outportb( 0x21, 0x00 );                     /* IRQs 0-7, active low */
        outportb( 0xA1, 0x00 );                     /* IRQs 8-15, active low */

	diags_subsys_stat( SYS_IRQ, DEV_SETUP );
    }
    else
    {
        /* disabling case */
        mobo_logf( LOG_INFO "Disabling ISA interrupts...\n");
        outportb( 0x21, 0xFF );                     /* IRQs 0-7, active low */
        outportb( 0xA1, 0xFF );                     /* IRQs 8-15, active low */
    }
}



/* Clear any pending interrupts (except machine checks) on this machine */


void plat_intclr( void )
{
    unsigned char iis0, iis1;
    int iters;
    int irq;
    int ipl;

    ipl = swpipl( 7 );                  /* noodling this stuff is dangerous */

#define MAX_ACKS        16

    /* until there are no more interrupts in service, send EOI signals */
    for ( iters=0; iters < MAX_ACKS; iters++)          /* 8 IRQ lines per reg */
    {
        irq = inIack();                         /* ack the int, returns IRQ */

        /* read Int in-service regs */
        outportb( 0x20, 0x0B );
        iis0 = inportb( 0x20 );
        outportb( 0xA0, 0x0B );
        iis1 = inportb( 0xA0 );

        if ( iis0 == 0 && iis0 == 0 )   break;

        if ( iis0 )      outportb( 0x20, 0x20 );                /* EOI */
        if ( iis1 )      outportb( 0xA0, 0x20 );                /* EOI */
    }

    swpipl( ipl );                      /* restore previous IPL */

    if ( iters >= MAX_ACKS )
        mobo_logf( LOG_CRIT __FUNCTION__
                   ": ERROR - recurring interrupts - not clearing\n");
}



/* Clear any pending system machine checks (NMIs) */

void plat_nmiclr( void )
{
    int val;

    /* try to de-assert in the southbridge */

    /* Technique 1: */
    /* by writing this back, we hope to clear any nasty error flags set */

    val = pcicfgrw( 0, PCI0_CYP_ISA, 0, 0x6 );
    pcicfgww( 0, PCI0_CYP_ISA, 0, 0x6, val );


    /* Technique 2: */
    /* read port 61. bit 7=1 on NMI.  bit 2 ->1->0 to reset NMI */

    val = inportb( 0x61 );                        /* NMI ctrl/status port */
    val |= 0x0C;
    outportb( 0x61, val );                        /* write 1's to clear NMIs */
    val &= ~0x0C;
    outportb( 0x61, val );                        /* write 0's to re-enable */


    /* now attempt clearing of the NXM error state by writing 1 to MISC:NXM */
    cchip[ MISC ].r = misc_m_nxm;               /* write NXM bit to clear */
    mb();                                       /* make sure it goes */
}




/*----------------------------------------------------------------------*/
/* Non-maskable interrupt (NMI) analysis */
/* NMI on Tsunami systems can come from a Pchip, or the southbridge, 
 * Or the halt button assertion.  Or a system management interrupt.
 */

void plat_mchkinterp( String interp, int scb, LogoutFrame_t *L)
{
    unsigned long tsu_dir0;

    switch (scb) {

    	case SCB_Q_SYS_EVENT:			/* a power management event */

	    /* Halt button assertion - a software halt event - on some early
	     * releases of UP2K/UP2K+ this switch was a toggle switch which 
	     * meant SRM could power up and get confused about the interrupt
	     * asserted.  Very duff.  Here we test for it (if we're passing 
	     * through) and give out some signals if we find halt asserted
	     */

	    /* Check ISUM to see if the halt button is asserted */
	    if ( L->ISUM & EV6__I_SUM__INT4 ) {
		up2k_halt_button_action( );
		mobo_logf( LOG_CRIT "halt button asserted\n" );
		sprintf_dbm( interp, "Halt button is asserted, please reset it "				 	"before continuing" );
	    }
	    break;

	case SCB_Q_SYSMCHK:

	    tsu_dir0 = cchip[ DIR0 ].r;		/* read the interrupt request */

	    if ( tsu_dir0 & PCI0_BUS_ERROR )
	    {
		pchip_err_analyse_clear( interp, 0 );
		break;
	    }

	    if ( tsu_dir0 & PCI1_BUS_ERROR )
	    {
		pchip_err_analyse_clear( interp, 1 );
		break;
	    }

	    if ( tsu_dir0 & IRQ_ERR_NXM )
	    {
		cchip_nxm_analyse_clear( interp );
		break;
	    }

	    if ( tsu_dir0 & PCI0_INT )
	    {
		sprintf_dbm( interp, "MACHINE CHECK?  ISA interrupt?\n"
				     "No further analysis available in diags "
				     MOBO_VERSION );
		break;
	    }
	    /* If we reached here we drop down into the no-match case */

	default:
	    sprintf_dbm( interp, "System Machine Check (NMI)\n"
				 "No further analysis available in diags "
				 MOBO_VERSION );
	    break;
    }

}


/*----------------------------------------------------------------------*/
/* SMP support */

/* This procedure is not well-documented, I lifted it from the DP264 hardware
 * reference manual and by reading the SROM sources... */

void plat_smpstart( unsigned long palbase )
{
    volatile unsigned long *t0 = (unsigned long *)(DBM_SUPER | 0x100UL);
    volatile unsigned long *t1 = (unsigned long *)(DBM_SUPER | 0x108UL);
    
    /* attempt to start the other CPU - whether it is 0 or 1 */
    *t0 = palbase;
    asm("mb");
    *t1 = palbase;
    asm("mb");

    tigwb( TIG_CPU0START, 0 );
    tigwb( TIG_CPU1START, 0 );
}



/*----------------------------------------------------------------------*/
/* I2C access functions */

#define NEEDS_CPU(x)		(1<<(x))
#define MAX_CPUID		1
#define NEEDS_SWPLUS		(1<<(MAX_CPUID+1))

/* Bus layout data */
const I2Cbus_t plat_i2c_bus[] = {

    /* Extra devices on UP2000+ are denoted so with 'NEEDS_SWPLUS' */

    /* Primary Module data */
    { I2C_SLOTB0_REV,	I2CSENS_EPROM,	I2CCHIP_PCF8582, SWORD_I2C_PCF8582_0,
		NEEDS_CPU(0),	"Primary Module asset info" },
    { I2C_SLOTB0_THERM,	I2CSENS_TEMP0,	I2CCHIP_LM75, SWORD_I2C_LM75_0,
		NEEDS_CPU(0),	"Primary CPU thermal sensor" },
    { I2C_SMD0,		I2CSENS_2_5V,	I2CCHIP_ADM9240, SWPLUS_I2C_ADM_0,
		NEEDS_SWPLUS | NEEDS_CPU(0),	"Primary SRAM supply level" },
    { I2C_SMD0,		I2CSENS_3_3V,	I2CCHIP_ADM9240, SWPLUS_I2C_ADM_0,
		NEEDS_SWPLUS,	"Primary 3.3V supply level" },
    { I2C_SMD0,		I2CSENS_5V,	I2CCHIP_ADM9240, SWPLUS_I2C_ADM_0,
		NEEDS_SWPLUS,	"Primary 5V supply level" },
    { I2C_SMD0,		I2CSENS_12V,	I2CCHIP_ADM9240, SWPLUS_I2C_ADM_0,
		NEEDS_SWPLUS,	"Primary 12V supply level" },

    /* Secondary Module data */
    { I2C_SLOTB1_REV,	I2CSENS_EPROM,	I2CCHIP_PCF8582, SWORD_I2C_PCF8582_1,
		NEEDS_CPU(1),	"Secondary Module asset info" },
    { I2C_SLOTB1_THERM,	I2CSENS_TEMP0,	I2CCHIP_LM75, SWORD_I2C_LM75_1,
		NEEDS_CPU(1),	"Secondary CPU thermal sensor" },
    { I2C_SMD1,		I2CSENS_2_5V,	I2CCHIP_ADM9240, SWPLUS_I2C_ADM_1,
		NEEDS_SWPLUS | NEEDS_CPU(1),	"Secondary SRAM supply level" },
    { I2C_SMD1,		I2CSENS_3_3V,	I2CCHIP_ADM9240, SWPLUS_I2C_ADM_1,
		NEEDS_SWPLUS,	"Secondary 3.3V supply level" },
    { I2C_SMD1,		I2CSENS_5V,	I2CCHIP_ADM9240, SWPLUS_I2C_ADM_1,
		NEEDS_SWPLUS,	"Secondary 5V supply level" },
    { I2C_SMD1,		I2CSENS_12V,	I2CCHIP_ADM9240, SWPLUS_I2C_ADM_1,
		NEEDS_SWPLUS,	"Secondary 12V supply level" },

    /* Motherboard and misc data */
    { I2C_MOBO_REV,	I2CSENS_EPROM,  I2CCHIP_PCF8582, SWORD_I2C_PCF8582_MB,
		0,	"Motherboard asset info" },
    { I2C_SMD0,		I2CSENS_TEMP0,	I2CCHIP_ADM9240, SWPLUS_I2C_ADM_0,
		NEEDS_SWPLUS,	"Motherboard thermal 0" },
    { I2C_SMD1,		I2CSENS_TEMP0,	I2CCHIP_ADM9240, SWPLUS_I2C_ADM_1,
		NEEDS_SWPLUS,	"Motherboard thermal 1" },
    { I2C_REG0,		I2CSENS_REGMUX,	I2CCHIP_PCF8574, SWPLUS_I2C_PCF8574_0,
		NEEDS_SWPLUS,	"System Control/Status 0" },
    { I2C_REG1,		I2CSENS_REGMUX,	I2CCHIP_PCF8574, SWPLUS_I2C_PCF8574_1,
		NEEDS_SWPLUS,	"System Control/Status 1" },
    { I2C_REG2,		I2CSENS_REGMUX,	I2CCHIP_PCF8574, SWPLUS_I2C_PCF8574_2,
		NEEDS_SWPLUS,	"Motherboard revision byte" },


    { I2C_NODEV, 0, 0, 0, 0, NULL }			/* End marker */
};


/* Platform-specific I2C initialisations */
DBM_STATUS plat_i2cinit( void )
{
    DBM_STATUS sval;
    int i;
    const I2Cbus_t *I;
    static const i2c_device lm_devs[] = { I2C_SLOTB0_THERM, I2C_SLOTB1_THERM };

    /* extra UP2000+ devices, unused on UP2000 */
    i2c_volt_t V = { 0, 0, 0, 0 };
    uint8 dummy;
    static const i2c_device adm_devs[] = { I2C_SMD0, I2C_SMD1 };

    diags_subsys_stat( SYS_I2C, DEV_PROBING );

    sval = i2cdev_init();
    if ( sval == STATUS_FAILURE )
    {
	mobo_logf( LOG_CRIT "platform I2C initialisation failed\n" );
	diags_subsys_stat( SYS_I2C, DEV_FAILED );
	return sval;
    }

    /* Are we on UP2000 or UP2000+?  This affects the devices available */
    /* Find out whether we can read a byte from anywhere in the ADM9240 */
    sval = i2cdev_read( SWPLUS_I2C_ADM_0, -1, 1, &dummy );
    if ( sval != STATUS_FAILURE )
	is_swplus = TRUE;		/* if its there, we're on UP2000+ */


    /* Configure LM75 thermal thresholds. */
    for ( i=0; i < ARRAYLEN( lm_devs ); i++ )
    {
	I = i2c_lookup( lm_devs[i] );
	if ( plat_i2c_select( I ) == STATUS_SUCCESS )
	{
	    sval = lm75_init( I->addr );
	    if ( sval == STATUS_FAILURE )
	    {
		mobo_logf( LOG_CRIT
			"Init failure with LM75 on Slot-B %d\n", i );
		mobo_alertf( "I2C Write Failure",
		    "Failure occurred whilst configuring LM75 thermal sensor\r"
		    "The sensor is located on Slot-B Module %d", i );
	    }
	}
    }


    if( is_swplus )		/* Extra devices for UP2000+ */
    {

	/* Bodge #1: Due to component shortages some UP2K+ systems ship with 
	 * PCF8574's instead of PCF8574A's.  This changes device addressing */
	/* To fix this up we scan for the devices at either address and choose
	 * the one that responds. */

	sval = i2cdev_read( SWPLUS_I2C_PCF8574_0, -1, 1, &dummy );
	if ( sval == STATUS_FAILURE )
	{
	    mobo_logf( LOG_INFO "I2C: "
		"Re-mapping PCF8574A to PCF8574 (0x%02X -> 0x%02X)\n",
		SWPLUS_I2C_PCF8574_0, SWPLUS_I2C_PCF8574_0_ALT );
	    I = i2c_lookup( I2C_REG0 );
	    I->addr = SWPLUS_I2C_PCF8574_0_ALT;
	}

	i2cdev_read( SWPLUS_I2C_PCF8574_1, -1, 1, &dummy );
	if ( sval == STATUS_FAILURE )
	{
	    mobo_logf( LOG_INFO "I2C: "
		"Re-mapping PCF8574A to PCF8574 (0x%02X -> 0x%02X)\n",
		SWPLUS_I2C_PCF8574_1, SWPLUS_I2C_PCF8574_1_ALT );
	    I = i2c_lookup( I2C_REG1 );
	    I->addr = SWPLUS_I2C_PCF8574_0_ALT;
	}

	sval = i2cdev_read( SWPLUS_I2C_PCF8574_2, -1, 1, &dummy );
	if ( sval == STATUS_FAILURE )
	{
	    mobo_logf( LOG_INFO "I2C: "
		"Re-mapping PCF8574A to PCF8574 (0x%02X -> 0x%02X)\n",
		SWPLUS_I2C_PCF8574_2, SWPLUS_I2C_PCF8574_2_ALT );
	    I = i2c_lookup( I2C_REG2 );
	    I->addr = SWPLUS_I2C_PCF8574_2_ALT;
	}


	/* Enable all our configured I2C ADM9240 sensors */
	for ( i=0; i < ARRAYLEN( adm_devs ); i++ )
	{
	    I = i2c_lookup( adm_devs[i] );
	    if ( plat_i2c_select( I ) == STATUS_SUCCESS )
	    {
		sval = adm_init( I->addr );
		if ( sval == STATUS_FAILURE )
		{
		    mobo_logf( LOG_CRIT
			"Write error to ADM9240 device %d in I2C init\n", i );
		    mobo_alertf( "I2C Write Failure",
			"Failure occurred writing to ADM9240 system monitor\r"
			"System health monitor %d failed initialisation", i );
		} else {
		    /* Extra device configuration data */
		    adm_add_dev( I->addr, I2CSENS_TEMP0, 
			TEMP_MAX_MOBO * I2C_TEMP_SCALE,
			TEMP_HYST_MOBO * I2C_TEMP_SCALE );

		    /* Note: we may be causing trouble as the sensor might 
		     * be reprogrammed below, which shouldn't be done...? */
		    adm_start( I->addr );
		}


		/* Bodge #2: The SRAM voltage level bits in I2C CSR0 aren't
		 * correctly set.  To fix this we assume it is only ever either
		 * 3.3V or 2.5V and choose the closest fit.
		 *
		 * As the cache supplies are hooked up to 2.5V, we only need to 
		 * reprogram the limits if they look more like 3.3V.
		 */

		adm_voltages( I->addr, I2CSENS_2_5V, &V );
		if ( (V.min < 25) && (V.v > ( (33 + 25) / 2 )) )
		{
		    mobo_logf( LOG_INFO "I2C: Reprogramming SRAM supply sensor "
			"from 2.5V to 3.3V on dev 0x%02X\n",
			I->addr );

		    /* Device is closer to 3.3V than 2.5V => reprogram limits */
		    /* NOTE: 3.3V is the max reading on a ADM9240 2.5V scale */
		    /* Use the overloaded semantic of '0' = max on range */
		    adm_add_dev( I->addr, I2CSENS_2_5V, 33 - 33 / 20, 0 );
		}
	    }   
	}
    }

    i2c_online = TRUE;
    diags_subsys_stat( SYS_I2C, DEV_SETUP );

    /* Log extra info supplied on UP2000+ */
    if ( is_swplus )
    {   
	/* There's a stack more info available on the I2C bus for UP2000+ */
	uint8 c0, c1, rev;

	info_submit( SRC_I2C, SUBJ_MOTHERBOARD, "Motherboard Variation",
		"UP2000+" );
	
	if ( i2cdev_read( SWPLUS_I2C_PCF8574_2, -1, 1, &rev ) == STATUS_FAILURE
	  || i2cdev_read( SWPLUS_I2C_PCF8574_1, -1, 1, &c1 ) == STATUS_FAILURE 
	  || i2cdev_read( SWPLUS_I2C_PCF8574_0, -1, 1, &c0 ) == STATUS_FAILURE )
	{   
	    mobo_logf( LOG_CRIT "I2C data could not be gathered!\n" );
	}
	else
	{
	    info_submit( SRC_I2C, SUBJ_MOTHERBOARD, "Motherboard Revision",
		    "Rev %02X", rev );

	    info_submit( SRC_I2C, SUBJ_CPU, "Processor module", "CPU 0: %s",
		    c0 & SWPL_R0_P0_PRESENT ? "Absent" : "Present" );
	    info_submit( SRC_I2C, SUBJ_CPU, "Processor module", "CPU 1: %s",
		    c0 & SWPL_R0_P1_PRESENT ? "Absent" : "Present" );


	    info_submit( SRC_I2C, SUBJ_CPU, "Processor core voltage", 
		    "CPU 0: %s", c0 & SWPL_R0_P0_CORE ? "Good" : "FAIL" );
	    info_submit( SRC_I2C, SUBJ_CPU, "Processor core voltage", 
		    "CPU 1: %s", c0 & SWPL_R0_P1_CORE ? "Good" : "FAIL" );
			

	    info_submit( SRC_I2C, SUBJ_CPU, "L2 Cache SRAM supply",
		    "CPU 0: %s", c0 & SWPL_R0_P0_SRAM ? "3.3V" : "2.5V" );
	    info_submit( SRC_I2C, SUBJ_CPU, "L2 Cache SRAM supply",
		    "CPU 1: %s", c0 & SWPL_R0_P1_SRAM ? "3.3V" : "2.5V" );


	    info_submit( SRC_I2C, SUBJ_MISC, "Power supply type",
		    c1 & SWPL_R1_RACKPSU ? "ATX PSU" : "Rack PSU" );

	    info_submit( SRC_I2C, SUBJ_MISC, "Power supply DC state", 
		    c1 & SWPL_R1_PWRGOOD ? "DC FAIL" : "DC Power Good" );

	    info_submit( SRC_I2C, SUBJ_MISC, "Power supply AC state",
		    c1 & SWPL_R1_ACFAIL ? "AC Power Good" : "AC FAIL" );


	    info_submit( SRC_I2C, SUBJ_CPU, "Processor type & speed match",
			"Type: %s, Speed: %s",
			c1 & SWPL_R1_TYPE_MATCH ? "OK" : "MISMATCH",
			c1 & SWPL_R1_SPEED_MATCH ? "OK" : "MISMATCH" );
	}
    }
    else
    {
	info_submit( SRC_I2C, SUBJ_MOTHERBOARD, "Motherboard Variation",
		"UP2000" );
    }

    return STATUS_SUCCESS;
}



/* Wire up the I2C bus (eg, by setting multiplexers) for this device */
/* For UP2000, we need to check for devices on in a module that is absent */

static BOOLEAN cpu_present( const int cpu )
{
    uint8 gpen;

    gpen = tigrb( (cpu == 0) ? TIG_GPEN5 : TIG_GPEN6 );
    return (gpen & 0x80) ? FALSE : TRUE;
}


DBM_STATUS plat_i2c_select( const I2Cbus_t *I )
{
    if ( I->config & NEEDS_CPU(0) && !cpu_present(0) )
		return STATUS_WARNING;

    if ( I->config & NEEDS_CPU(1) && !cpu_present(1) )
		return STATUS_WARNING;

    if ( I->config & NEEDS_SWPLUS && !is_swplus )
		return STATUS_WARNING;

    return STATUS_SUCCESS;
}




/*----------------------------------------------------------------------*/
/* Debugging LED writes */

#define LEDPORT 0x80

void outled( unsigned d)
{
    outportb(LEDPORT, d);
}


/*----------------------------------------------------------------------*/
/* Magic widgetry */

/* We can establish if the FSB jumper is set by looking at bit 0 =0 for FSB */

unsigned char up2k_get_behav_code( void )
{
    unsigned gpen4;

    gpen4 = tigrb( TIG_GPEN4 );
    if ( (gpen4 & 0x1) == 0 )	return BIOSRECOVER;	/* FSB pin in place */
    else			return UNSPECIFIED;	/* something else */
}



/*------------------------------------------------------------------------*/
/* Report halt button assertion: design flaw in early UP2K+ */

static void up2k_halt_button_action( void )
{
    BeepCode( beep_k_haltbutton );
    bring_up_console( );
    mobo_alertf( "Halt button is asserted",
	 "Your halt button is active, please release it\r"
	 "and then press a key to continue the boot process");
}


/*------------------------------------------------------------------------*/
/* Sanity-check the SROM data and jumper settings */
/* Both processors *should* have identical settings for clock speed 
 * and bcache.  We also compare processor speed with SROM cycle frequency.
 */
 

static DBM_STATUS up2k_jumper_sanity( void )
{
    unsigned clk0, clk1, bcache0, bcache1, present0, present1;
    unsigned char gpen5, gpen6;
    unsigned long srom_cycle, actual_cycle;
    unsigned cycle_dev;
    DBM_STATUS rval = STATUS_SUCCESS;

    gpen5 = tigrb( TIG_GPEN5 );
    gpen6 = tigrb( TIG_GPEN6 );
    present0 = gpen5 & 0x80 ? 0 : 1;		/* bit 7=0 -> present */
    present1 = gpen6 & 0x80 ? 0 : 1;		/* bit 7=0 -> present */
    clk0 = gpen5 & 0x7U;
    clk1 = gpen6 & 0x7U;
    bcache0 = (gpen5 >> 3) & 0x7U;
    bcache1 = (gpen6 >> 3) & 0x7U;

    /* Check both processors are matched */
    if( present0 && present1 )
    {
	/* check processor speeds are matched */
	if ( clk0 != clk1 )
	{
	    BeepCode( beep_k_hw_mismatch );
	    bring_up_console( );
	    mobo_alertf( "Processor speed jumper mismatch",
		"Your motherboard is not configured with both processors\r"
		"at the same cycle frequency.  This is unsupported!" );
	    rval = STATUS_FAILURE;
	}


	/* Check that bcache sizes match */
	if ( bcache0 != bcache1 )
	{
	    BeepCode( beep_k_hw_mismatch );
	    bring_up_console( );
	    mobo_alertf( "Processor cache size jumper mismatch",
		"Your motherboard is configured with both processors\r"
		"having different B-cache sizes.  This is unsupported!" );
	    rval = STATUS_FAILURE;
	}
    }


    /* Check that the processor speed matches the SROM cycle count */
    srom_cycle = 1000000000000UL / primary_impure->CYCLE_CNT;
    switch ( clk0 )
    {
	case 5:			/* 667 MHz */
	    actual_cycle = 666666666UL;
	    break;

	default:
	    /* don't know any other speed codes at the moment */
	    actual_cycle = 0UL;
	    break;
    }
    if ( actual_cycle != 0 )
    {
	cycle_dev = ( actual_cycle * 100 ) / srom_cycle;
	if ( cycle_dev < 95 || cycle_dev > 105 )
	{
	    BeepCode( beep_k_srom_miscalc );
	    bring_up_console( );
	    mobo_alertf( "SROM miscalculation of processor speed",
		"The SROM firmware has miscalculated the "
		    "processor cycle frequency\r"
		"according to the motherboard's defined configuration" );
	    rval = STATUS_FAILURE;
	}
    }
    return rval;
}

/*------------------------------------------------------------------------*/
/* Monitor the periodic timer and scan for accuracy/drift of the clock */

/* Home in on an integer square root using bisection */
static unsigned long isqrt(unsigned long val)
{
    unsigned long upper = val / 2;
    unsigned long lower = 2;
    unsigned long tryval, tryresult;

    do {
	tryval = (upper - lower) / 2 + lower;
	tryresult = tryval * tryval;
	if (tryresult == val)
	    return tryval;
	if (tryresult > val)
	    upper = tryval;
	if (tryresult < val)
	    lower = tryval;
    } while (upper - lower > 1);

    return (upper * upper - val > val - lower * lower) ? lower : upper;
}


static double ticks2usecs( unsigned long ticks )
{
    return (double)(ticks * primary_impure->CYCLE_CNT) * 1.0e-6;
}


#define TEST_PASSES 102

double up2k_rtc_xbar=0.0, up2k_rtc_sigma=0.0;
const double up2k_rtc_mu = 1.0e6 / (double) HZ;

DBM_STATUS up2k_check_rtc(void)
{
    const unsigned n = TEST_PASSES - 2;
    int mean, variance, stdev;
    unsigned i;
    int ipl;
    unsigned long start, end, ticks, sum, sumsq, min, max;
    const double sigma_threshold = 3.0;
    unsigned long cycles[TEST_PASSES];

    up2k_rtc_xbar = up2k_rtc_sigma = 0.0;


    /* note, in the below we're faking reading of interrupts since jiffies
     * does weird things in the loop below */

    ipl = swpipl(7);

    /* Set the RTC to the expected mode */
    cmoswb(CM_STATUSA, CM_STATUSA__INIT);
    cmoswb(CM_STATUSB, CM_STATUSB__INIT | STATUSB_PER);

    /* Before we do anything, verify that the timer is ticking */
    cmosrb(CM_STATUSC);
    start = rpcc();
    while ((cmosrb(CM_STATUSC) & STATUSC_PIT) == 0) {
	end = rpcc();
	if (end <= start)
	    end += 1UL << 32;
	ticks = end - start;
	if (ticks >= 1UL << 30)		/* a sec or two */
	    return RTC_STOPPED;
    }

    /* scan all the array to ensure we don't take any page faults */
    memset(cycles, 0, TEST_PASSES * sizeof(cycles[0]));


    /* collect our measurements, comparing processor cycles with timer period */
    for (i = 0; i < TEST_PASSES; i++) {
	while ((cmosrb(CM_STATUSC) & STATUSC_IRQ) == 0)
	    continue;

	cycles[i] = rpcc();
    }

    /* If periodic interrupt wasn't enabled by default, disable it here */
    cmoswb(CM_STATUSB, CM_STATUSB__INIT);
    swpipl(ipl);


    /* We discard the first data point, which is an incomplete timer period */
    /* ... and in finding the time differences we lose the second data point */
    /* Implementation note: elapsed ticks will be approx 1,000,000, so sumsq
     * will be seriously large - 10^12 - fortunately still within limits of 
     * 64-bit integer arithmetic */

    for (i = 2, sum = 0, sumsq = 0, max = 0, min = ~0UL; i < TEST_PASSES;
	 i++) {
	/* counter wraparound */
	if (cycles[i] < cycles[i - 1])
	    cycles[i] += 1UL << 32;
	ticks = cycles[i] - cycles[i - 1];
	sum += ticks;
	sumsq += ticks * ticks;
	if (ticks < min)
	    min = ticks;
	if (ticks > max)
	    max = ticks;
    }

    mean = sum / n;
    variance = sumsq / n - mean * mean;
    stdev = isqrt(variance);

    up2k_rtc_xbar = ticks2usecs(mean);
    up2k_rtc_sigma = ticks2usecs(stdev);


    /* Now do our error analysis.  Reject the timer system if:
     *
     * 1) The sample standard deviation (sigma) is greater than 3us (say)
     * 2) The sample mean is greater than 2*sigma from underlying mean
     */

    if (up2k_rtc_xbar - 2.0 * up2k_rtc_sigma > up2k_rtc_mu)
	return RTC_SLOW;

    if (up2k_rtc_xbar + 2.0 * up2k_rtc_sigma < up2k_rtc_mu)
	return RTC_FAST;

    if (up2k_rtc_sigma > sigma_threshold)
	return RTC_ERRATIC;

    return RTC_OK;
}


DBM_STATUS up2k_rtc_analyse(void)
{
    int rval;
    rval = up2k_check_rtc();

    if (rval != RTC_OK)
    {
	/* It's possible we got here because we've just powered on
	 * and the timer has not yet stabilised.
	 * Wait a while and get a second opinion.
	 */

	sleep( 1 );
	rval = up2k_check_rtc();

	if ( rval != RTC_OK )
	{
	    /* bring up a console, alert the user */
	    BeepCode( beep_k_rtc_duff );
	    bring_up_console( );
	}
    }

    switch (rval) {

    case RTC_OK:
	mobo_logf(LOG_INFO
	    "Estimated timer period %0.02fus (want %0.02fus), sigma=%0.02fus\n",
	    up2k_rtc_xbar, up2k_rtc_mu, up2k_rtc_sigma);
	break;

    case RTC_ERRATIC:
	mobo_logf(LOG_CRIT
	    "Timer too erratic: measured %0.02fus, sigma=%0.02fus, want %0.02fus\n",
	    up2k_rtc_xbar, up2k_rtc_sigma, up2k_rtc_mu);
	mobo_alertf( "System timer is drifting",
		"The system timer was measured to be inaccurate beyond\r"
		"acceptable thresholds (drift deviation was %0.02fus)",
		up2k_rtc_sigma );
	break;

    case RTC_FAST:
	mobo_logf(LOG_CRIT
	    "Timer is too fast: RTC period measured at %0.02fus, want %0.02fus\n",
	    up2k_rtc_xbar, up2k_rtc_mu);
	mobo_alertf( "System timer is running fast",
		"The system timer was measured to be fast beyond acceptable\r"
		"thresholds (timer period was %0.02fus, ideal is %0.02fus)",
		up2k_rtc_xbar, up2k_rtc_mu );
	break;

    case RTC_SLOW:
	mobo_logf(LOG_CRIT
	    "Timer is too slow: RTC period measured at %0.02fus, want %0.02fus\n",
	    up2k_rtc_xbar, up2k_rtc_mu);
	mobo_alertf( "System timer is running slow",
		"The system timer was measured to be slow beyond acceptable\r"
		"thresholds (timer period was %0.02fus, ideal is %0.02fus)",
		up2k_rtc_xbar, up2k_rtc_mu );
	break;

    case RTC_STOPPED:
	mobo_logf(LOG_CRIT "Timer is not ticking!\n");
	mobo_alertf( "System timer is not working",
		"No timer interrupts were detected.\r"
		"This board is not operational." );
	break;

    default:
	break;
    }

    return ( rval == RTC_OK ) ? STATUS_SUCCESS : STATUS_FAILURE;
}

