/* Although a rectangle is a polygon, it's important enough
 to be a descendant of GraphicsObject in its own right.  It
 has two descendants, RndRect and Ellipse.

 Rectangles are defined by four numbers.  The first two are
 the x and y coordinates, respectively, of the upper left
 hand corner, called the origin.  The last two are the x and
 y coordinates of the lower right hand corner, called the
 far corner or sometimes just the corner.

 You can make a literal Rect with the & character.  For
 example, the following defines a literal Rect with origin
 at (3,5) and corner at (8,9): &(3 5 8 9).  */!!

/* inherit(Polygon, #Rect, nil, nil, nil) */ !!

now(class(Rect)) !!

/* Return a new Struct with 8 elements,
  the size of an MS-Windows rectangle.  */
Def new(self)
{ ^variableNew(self:Behavior, 8);
}!!


now(Rect) !!

/* Expand the rectangle by x and y units
  (subtracts x from left, adds x to right,
  subtracts y from top, and adds y to
  bottom).  Directly alter the Rect
  object and return it.  */
 Def inflate(self, x, y)
{ Call InflateRect(self, x, y)
}!!

/* Move the rectangle by x and y units
  (adds x to left and right, and adds y to
  top and bottom).  Directly alter the
  Rect object and return it.  */
 Def offset(self, x, y)
{ Call OffsetRect(self, x, y)
}!!


/* Draw the Rect using the specified
  handle to a display context. */
 Def draw(self, hdc)
{       Call Rectangle(hdc, left(self),
  top(self), right(self), bottom(self))
}!!

/* Set the left of the Rect (x value
  of the origin) to newValue.  */
Def setLeft(self, newValue)
{ putWord(self:Struct, newValue, 0)
}!!

/* Set the right of the Rect (x value of
  the far corner) to newValue.  */
Def setRight(self, newValue)
{ putWord(self:Struct, newValue, 4)
}!!

/* Set the top of the Rect (y value of
  the origin) to newValue.  */
Def setTop(self, newValue)
{ putWord(self:Struct, newValue, 2)
}!!


/* Set the bottom of the Rect (y value
  of the far corner) to newValue.  */
Def setBottom(self, newValue)
{ putWord(self:Struct, newValue, 6)
}!!

/* Set the origin to aPoint.  For
  example, setOrigin(&(0 0 0 0),3@4)
  returns Rect(3L 4L 0L 0L).  */
 Def setOrigin(self, aPoint)
{ setLeft(self, aPoint.x);
  setTop(self, aPoint.y)
}!!


/* Set the far corner to aPoint.  For
  example, setCorner(&(0 0 0 0),3@4)
  returns Rect(0L 0L 3L 4L).  */
 Def setCorner(self, aPoint)
{ setRight(self, aPoint.x);
  setBottom(self, aPoint.y)
}!!

/* Return the width of the receiver
  (right - left).  */
Def width(self)
{ ^abs(right(self) - left(self))
}!!


/* Return the height of the receiver
  (Bottom - top).  */
Def height(self)
{ ^abs(bottom(self) - top(self))
}!!


/* Fill self with the given brush pattern,
  using the specified handle to a display
  context. */
 Def fill(self, aBrush, hdc)
{ Call FillRect(hdc, self, aBrush)
}!!


/* Return the x coordinate of the
  origin of the Rect object.  For example,
  left(&(10 20 30 40)) returns 10.  */
Def left(self)
{ ^asInt(wordAt(self, 0));
}!!

/* Return the y coordinate of the
  origin of the Rect object.  For example,
  top(&(10 20 30 40)) returns 20.  */
Def top(self)
{ ^asInt(wordAt(self, 2));
}!!



/* Return the x coordinate of the far
  corner of the Rect object.  For example,
  right(&(10 20 30 40)) returns 30.  */
 Def right(self)
{ ^asInt(wordAt(self, 4));
}!!



/* Return the y coordinate of the far
  corner of the Rect object.  For example,
  bottom(&(10 20 30 40)) returns 40.  */
Def bottom(self)
{ ^asInt(wordAt(self, 6));
}!!


/* Intialize the left, top, right, and
  bottom of the Rect object.  For instance,
  if Joe is a Rect object, then init(Joe,1,
  2,3,4) returns Rect(1L 2L 3L 4L). */
 Prim init(self, left, top, right, bottom):Rect!!
