/* A SortedCollection is an indexed collection whose elements
 are in some kind of sorted order, such as ascending or
 descending.  Whenever a new element is added, it is placed
 in the collection such that sorted order is maintained.

 The order in which the collection is sorted is completely
 arbitrary, and is determined by an instance variable called
 compareBlock.  */!!

inherit(OrderedCollection, #SortedCollection,
  #(compareBlock /* A two-argument block which compares the two arguments */),
  2, 1)!!

now(class(SortedCollection));!!

now(SortedCollection);!!

/* Initialize the SortedCollection object.
  By default, the compareBlock is set so that
  the elements are sorted in ascending order.
  lastElement and firstElement are also set
  equal to zero.  */
Def init(self)
{   compareBlock := { using(item1,item2) item1 < item2 };
    firstElement := lastElement := 0;
}!!

/* Create a new collection, preserves the old compareBlock,
  copy all the elements from the old collection to the new,
  and swap object pointers so that self refers to the newly
  created collection.  */
Def grow(self | newColl)
{  newColl := new(SortedCollection, limit(self) + 8);
   newColl.compareBlock := compareBlock;
   do(self,
          { using(element) add(newColl, element)
          });
   swap(newColl,self);
}!!

/* Re-sort self according to newCompareBlock,
  return self. */
Def setCompareBlock(self, newCompareBlock | coll)
{   coll := new(SortedCollection, size(self));
    coll.compareBlock := newCompareBlock;
    do(self,
          { using(elem) add(coll, elem)
          });
    swap(coll, self);
}!!

/* Search for an target in the receiver.
  Utilize a binary search, since we are
  searching for an item in a sorted list.
  Return a two-element tuple where the first
  element is a boolean flag indicating whether
  or not the target was found.  The second
  element of the tuple can mean two different
  things.  If the target was found, the
  second element of the tuple is the index
  at which the target was found.  If it
  wasn't, the second element is the index
  at which the target should be inserted.  */
Def findItemIndex(self, target | top,bot,mid)
  {
  if (lastElement <= firstElement)
  then  firstElement := lastElement := 0;
    ^tuple(nil,0);
  endif;
  if not(eval(compareBlock,target,
    at(self:Object,lastElement-1)))
  then ^tuple(nil,lastElement);
  endif;
  bot := firstElement;
  top := lastElement-1;
  loop
  while (top > bot)
  begin   mid := (top+bot)/2;
    if (target = at(self:Object, mid))
    then  ^tuple(true, mid);
    else
      if eval(compareBlock,target,
        at(self:Object, mid))
      then   top := mid;
      else   bot := mid + 1;
      endif;
    endif;
  endLoop;
  ^tuple(nil, top);
  }!!

/*  Add an item to a sorted collection.  First
  determine where it should go and then puts the
  item there. */
Def add(self, item | index, elem)
  {
  if (index := lastElement) == limit(self)
  then grow(self);
  endif;
  lastElement := lastElement + 1;
  loop
  while index > firstElement
  begin elem := at(self:Object, index - 1);
    if not(eval(compareBlock, item, elem))
    then ^put(self:Object, item, index);
    endif;
    put(self:Object, elem, index);
  index := index - 1;
  endLoop;
  ^put(self:Object, item, firstElement);
  }!!


/* Remove the specified item from a
  sorted collection.  */
Def remove(self,item | foundIdxTuple)
{  foundIdxTuple := findItemIndex(self,item);
   if foundIdxTuple[0]
   then
     remove(self:OrderedCollection,foundIdxTuple[1]);
   else
     error(self,stackTop(0),#elemNotFndError);
   endif;
}!!
