PAGE	60, 132

;------------------------------------------------------------------
;
;	KERNEL.ASM: System management kernel for the SL architecture.
;
;
;	This program demonstrates the usage of the SL architecture's
;	system management capabilities for power management. 
;
;	Author:	Desmond Yuen
;
;	Date:	Dec 19, 1992
;
;	Revision History:	
;
;	Copyright 1992 by McGraw-Hill, Inc. All rights reserved.
;
;------------------------------------------------------------------


;---------------------------------------------------------------------
;
;	This program was tested on the Intel386(tm) SL Microprocessor
;	evaluation board. The same program should work on most SL based 
;	systems with slight modifications.
;
;----------------------------------------------------------------------


;------------------------------------------------------------------
;
;		Include files
;
;------------------------------------------------------------------
Include	superset.inc

code	segment	byte	PUBLIC	'code'
	assume	cs:code,ds:code,es:code,ss:code  
	ORG	0
ORIGIN	EQU	$

.386p

start:	
        jmp	kernel
	db	"System Management Kernel V1.00", 00


;------------------------------------------------------------------
;
;		Equates
;
;------------------------------------------------------------------



IFDEF	DEBUG

cr	EQU	0AH		; used in debugging messages
lf	EQU	0DH

ENDIF


EXTRN	open_cpupwrmode:near, close_cpupwrmode:near, open_ibu:near
EXTRN	open_omcu:near, open_ebu:near, open_ccu:near, close_386sl:near
EXTRN	open_360sl:near, read_360sl:near, write_360sl:near, close_360sl:near
EXTRN   exec_ltrp:near, exec_lstdby:near, suspend:near, resume:near
EXTRN	devices_off:near, devices_on:near, tst_buf:near

; defines stack for the SMM kernel

Stack		dw	2000	dup (?)
Topofstack	label   word

; buffers for saving CPU registers

		dw      100h	dup (?) 
		cpu_state       label   word

cr2_buf         dd      0
dr0_buf         dd      0
dr1_buf         dd      0
dr2_buf         dd      0
dr3_buf         dd      0
tr6_buf         dd      0
tr7_buf		dd	0

sm_cntrl	db	0
a20_state	db	0

;------------------------------------------------------------------
;
;		Main program begins here
;
;------------------------------------------------------------------

REQTAB	LABEL	WORD
INIT	dw      0	; 0 - initialization


;------------------------------------------------------------------
;
;		SM-RAM Entry Point
;
;------------------------------------------------------------------

	


;-----------------------------------------------------------------
;
;	Kernel Routine
;	
;	Algorithm: Determine what caused the SMI by reading
;	the status registers and pass control to the system 
;	management request handler	
;
;-----------------------------------------------------------------

kernel:

; determines what caused an SMI

        cli                     
	mov	ax, cs		; initialize segment registers
	mov	ds, ax
	mov	es, ax
        mov     ss, ax
	mov	sp, offset Topofstack

        mov     al, 0E1H	; SMM entry post code
        out     80h, al

	xor	cx, cx
	loop	$

;----------------------------------------------------------------
;
; test for resume request
;
;----------------------------------------------------------------


	mov	al, 07fh	; check suspend flag in XCMOS RAM
	out	XCMOS_INDEX, al
	jmp	$+2
	jmp	$+2
	in	al, XCMOS_DATA

	and	al, 08h		; is SUS_STAT bit set
	jz	not_resume	; not a resume request

	call	resume_sys
	jmp	exit_smi


;------------------------------------------------------------------
;
;		SMI request processing jump table
;
;------------------------------------------------------------------



not_resume:

        mov     al, 0E2H	; normal SMI request post code
        out     80h, al

	xor	cx, cx
	loop	$

	call	open_360sl
	mov	bl, SM_REQ_STS
	call	read_360sl	; read SM request reg
        call    close_360sl

	shl	al, 1		; is SMI caused by external SMI
	shl	al, 1
 	jnc	ltrp 
	call	exec_ext_SMI

ltrp:	shl	al, 1		; is SMI caused by local trap request
        shl     al, 1
	jnc	lstd		; no
	call	exec_ltrp

lstd:	shl	al, 1		; is SMI caused by local standby request
	jnc	gstd		; no
	call	exec_lstdby

gstd:	shl	al, 1		; is SMI caused by global standby request
	jnc	sus
	call	exec_gstdby
        
sus:	shl	al, 1		; is SMI caused by hardware suspend request
	jnc	soft
	call	exec_suspend

soft:	shl	al, 1
	jnc	exit_smi
	call	exec_soft_SMI	; SMI caused by software SMI request

exit_smi:

; terminate SMI service routine

	call	open_360sl
        xor	bh, bh
	mov	bl, SMI_CLR
	call	write_360sl   	; re-enable SMI
	mov	bl, SMI_MARK
	call	write_360sl	; re-enable CPU reset
	call	close_360sl

	db	0Fh, 0AAh	; execute RSM instr


;------------------------------------------------------------------
;
;		System Management Request Handler
;
;------------------------------------------------------------------


;------------------------------------------------------------------
;
;	exec_ext_smi: Procedure to service external SMI request
;
;	Input:	None
;
;	Output:	None
;
;------------------------------------------------------------------


exec_ext_smi	proc	near	; external smi request

        mov     al, 0e3h	; ext SMI post code	
        out     80h, al	

        call    open_360sl
        mov     bl, SM_REQ_STS  ; clear extsmi request bit
        call    read_360sl
        and     al, 0bfh
        mov     bh, al
        call    write_360sl
        call    close_360sl
	ret
exec_ext_smi	endp

;------------------------------------------------------------------
;
;	exec_soft_smi: Procedure to service software SMI request
;
;	Input:	None
;
;	Output:	None
;
;------------------------------------------------------------------


exec_soft_smi	proc	near	; software smi request

        mov     al, 0e4h	; ext SMI post code	
        out     80h, al	

        call    open_360sl
        mov     bl, SM_REQ_STS  ; clear software request bit
        call    read_360sl
        and     al, 0feh
        mov     bh, al
        call    write_360sl
        call    close_360sl
	ret
exec_soft_smi	endp


;------------------------------------------------------------------
;
;	exec_gstdby: Procedure to put system in and out of global standby
;
;	Input:	None
;
;	Output:	None
;
;------------------------------------------------------------------



exec_gstdby	proc	near	; global standy request

        push    ax

	mov	al, 0e6h	; gstdby post code
	out	80h, al

        call    open_360sl
        mov     bl, SM_REQ_STS  ; clear global standby request
        call    read_360sl
	and	al, 0fbh
	or	al, 80h		; set SYS_IN_STDBY bit
        mov     bh, al
        call    write_360sl

        mov     bl, SYS_EVNT_CFG2       ; disable system events
        call    read_360sl
        and     al, 0feh
        mov     bh, al
        call    write_360sl

        mov     bl, STP_BRK_CFG2        ; enable stop break event
        call    read_360sl
        or      al, 01h
        mov     bh, al
        call    write_360sl
        call    close_360sl

	call	devices_off	; turn off some devices to save power

; stop CPU clock

        call    open_360sl

        mov     bl, STP_CLK
        call    read_360sl

	xor	cx, cx		; delay to let stop clock takes effect
	loop	$

        mov     bl, SM_REQ_STS  ; clear SYS_IN_STDBY bit
        call    read_360sl
	and	al, 07fh
        mov     bh, al
        call    write_360sl

        mov     bl, SYS_EVNT_CFG2       ; re-enable system events
        call    read_360sl
        or      al, 01h
        mov     bh, al
        call    write_360sl

        mov     bl, STP_BRK_CFG2        ; disable stop break event
        call    read_360sl
        and     al, 0feh
        mov     bh, al
        call    write_360sl
        call    close_360sl

	call	devices_on	; re-enable powered off devices
        pop     ax

	ret
exec_gstdby	endp	

;------------------------------------------------------------------
;
;	exec_suspend: Procedure to suspend system
;
;	Input:	None
;
;	Output:	None
;
;------------------------------------------------------------------


exec_suspend	proc	near	; suspend system request

	mov	al, 0e7h	; suspend request post code
	out	80h, al

	call	save_a20	; save A20 gate
	call	save_cpu	; save CPU registers

; find source of suspend request

        call    open_360sl
	mov	bl, SM_REQ_CNTRL
	call	read_360sl
	mov	byte ptr [sm_cntrl], al	; save state of SM_REQ_CNTRL reg
        call    close_360sl
	call	exec_srbtn_sus
	ret
exec_suspend	endp	



;------------------------------------------------------------------
;
;	resume_sys: Procedure to restore system to state prior to
;		    suspend
;
;	Input:	None
;
;	Output:	None
;
;------------------------------------------------------------------


resume_sys	proc	near	; resume system request

	call	resume
	call	rstor_cpu	; restore cpu state
	call	rstor_a20	; restore a20 gate
	call	open_360sl

	mov	al, byte ptr [sm_cntrl]	; restore state of SM_REQ_CNTRL reg
	mov	bh, al
	mov	bl, SM_REQ_CNTRL
	call	write_360sl	; re-enable SMIs

        mov     bl, SPND_STS	; clear suspend request bit
        xor     bh, bh
        call    write_360sl

        mov     bl, SM_REQ_STS	; clear suspend request bit
        xor     bh, bh
        call    write_360sl
	call	close_360sl
	ret
resume_sys	endp


;------------------------------------------------------------------
;
;	exec_srbtn_sus: Procedure to handle suspend resume button
;			suspend request
;
;	Input:	None
;
;	Output:	None
;
;------------------------------------------------------------------


exec_srbtn_sus	proc	near
	call	suspend		; put system in suspend
        ret
exec_srbtn_sus	endp


;------------------------------------------------------------------
;
;	rstor_cpu: Procedure to save CPU state before suspend
;
;	Input:	None
;
;	Output:	None
;
;------------------------------------------------------------------


save_cpu proc near

; save registers not saved by SMI microcode

	mov	eax, cr2
	mov	dword ptr [cr2_buf], eax
	mov	eax, dr0
	mov	dword ptr [dr0_buf], eax
	mov	eax, dr1
	mov	dword ptr [dr1_buf], eax
	mov	eax, dr2
	mov	dword ptr [dr2_buf], eax
	mov	eax, dr3
	mov	dword ptr [dr3_buf], eax
	mov	eax, tr6
	mov	dword ptr [tr6_buf], eax
	mov	eax, tr7
	mov	dword ptr [tr7_buf], eax

; CPU core dump will be overwritten by the resume smi
; so, save CPU core dump before suspend

	mov	si,0ffffh
	mov	di,offset cpu_state
	std
	mov	cx,100h
	rep	movsw
	cld

	ret
save_cpu endp


;------------------------------------------------------------------
;
;	rstor_cpu: Procedure to restore CPU state after resume
;
;	Input:	None
;
;	Output:	None
;
;------------------------------------------------------------------

     
rstor_cpu proc

; restore registers not saved by SMI microcode

	mov	eax, dword ptr [cr2_buf]
	mov	cr2, eax
	mov	eax, dword ptr [dr0_buf]
	mov	dr0, eax
	mov	eax, dword ptr [dr1_buf]
	mov	dr1, eax
	mov	eax, dword ptr [dr2_buf]
	mov	dr2, eax
	mov	eax, dword ptr [dr3_buf]
	mov	dr3, eax
	mov	eax, dword ptr [tr6_buf]
	mov	tr6, eax
	mov	dword ptr [tr7_buf], eax
	mov	tr7, eax

; CPU core dump will be overwritten by the resume smi
; so, save CPU core dump before suspend

	mov	di,0ffffh
	mov	si,offset cpu_state
	std
	mov	cx,100h
	rep	movsw
	cld

	ret
rstor_cpu endp


;------------------------------------------------------------------
;
;	save_a20: Procedure to save A20 gate status before suspend
;
;	Input:	None
;
;	Output:	None
;
;------------------------------------------------------------------


save_a20 proc near

	mov	ah, 01h		; test output buffer
	call	tst_buf

	mov	al,0D0h		; read output port
	out	64h,al
	jmp	$+2
	jmp	$+2
	mov	ah, 02h		; test input buffer
	call	tst_buf

	in	al,60h		
	mov	byte ptr [a20_state], al	; save a20 gate
	
	ret
save_a20 endp


;------------------------------------------------------------------
;
;	rstor_a20: Procedure to restore A20 gate status after resume 
;
;	Input:	None
;
;	Output:	None
;
;------------------------------------------------------------------


     
rstor_a20 proc

	mov	ah, 01h		; test output buffer
	call	tst_buf

	mov	al,0D1h
	out	64h,al
	jmp	$+2
	jmp	$+2
	mov	ah, 02h		; test input buffer
	call	tst_buf
	mov	al,byte ptr [a20_state]
	or	al, 01h
	out	60h,al

	ret
rstor_a20 endp

;-------------------------------------------------------------------
;
;		END OF SYSTEM MANAGEMENT PROGRAM
;
;-------------------------------------------------------------------


code	ends
	end



